<?php
// admin/class-unify-admin-ajax.php

defined('ABSPATH') || exit;

class Unify_Admin_Ajax {
    
    public static function init() {
        add_action('wp_ajax_unify_save_banner_settings', [__CLASS__, 'save_banner_settings']);
        add_action('wp_ajax_unify_update_dsar_status', [__CLASS__, 'update_dsar_status']);
        add_action('wp_ajax_unify_save_dsar_settings', [__CLASS__, 'save_dsar_settings']);
        add_action('wp_ajax_unify_save_ai_system', [__CLASS__, 'save_ai_system']);
        add_action('wp_ajax_unify_get_ai_system', [__CLASS__, 'get_ai_system']);
        add_action('wp_ajax_unify_delete_ai_system', [__CLASS__, 'delete_ai_system']);
        add_action('wp_ajax_unify_run_ai_scan', [__CLASS__, 'run_ai_scan']);
        add_action('wp_ajax_unify_preview_transparency_card', [__CLASS__, 'preview_transparency_card']);
        add_action('wp_ajax_unify_save_nis2_settings', [__CLASS__, 'save_nis2_settings']);
        add_action('wp_ajax_unify_export_nis2_logs', [__CLASS__, 'export_nis2_logs']);
        
        // Policy Management
        add_action('wp_ajax_unify_save_policy', [__CLASS__, 'save_policy']);
        add_action('wp_ajax_unify_get_policy', [__CLASS__, 'get_policy']);
        add_action('wp_ajax_unify_delete_policy', [__CLASS__, 'delete_policy']);
        
        // Vulnerability Scanner
        add_action('wp_ajax_unify_run_vuln_scan', [__CLASS__, 'run_vuln_scan']);
        add_action('wp_ajax_unify_save_manual_vulnerability', [__CLASS__, 'save_manual_vulnerability']);
        add_action('wp_ajax_unify_get_manual_vulnerabilities', [__CLASS__, 'get_manual_vulnerabilities']);
        add_action( 'wp_ajax_unify_delete_manual_vulnerability', array( __CLASS__, 'delete_manual_vulnerability' ) );
        add_action( 'wp_ajax_unify_export_vuln_report', array( __CLASS__, 'export_vuln_report' ) );
        add_action( 'wp_ajax_unify_save_modules', array( __CLASS__, 'save_modules' ) );
        
        // Reports
        add_action( 'wp_ajax_unify_get_consent_stats', array( __CLASS__, 'get_consent_stats' ) );
        add_action( 'wp_ajax_unify_get_overview_stats', array( __CLASS__, 'get_overview_stats' ) );
        add_action( 'wp_ajax_unify_get_dsar_stats', array( __CLASS__, 'get_dsar_stats' ) );
        add_action( 'wp_ajax_unify_get_security_stats', array( __CLASS__, 'get_security_stats' ) );
        add_action( 'wp_ajax_unify_export_report_pdf', array( __CLASS__, 'export_report_pdf' ) );
        
        // Dashboard
        add_action( 'wp_ajax_unify_toggle_checklist_item', array( __CLASS__, 'toggle_checklist_item' ) );

        // License
        add_action( 'wp_ajax_unify_activate_license', array( __CLASS__, 'activate_license' ) );
        add_action( 'wp_ajax_unify_deactivate_license', array( __CLASS__, 'deactivate_license' ) );
        add_action( 'wp_ajax_unify_save_general_settings', array( __CLASS__, 'save_general_settings' ) );
    }
    
    public static function save_banner_settings() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }
        
        // List of fields to save
        $fields = [
            'unify_banner_position' => 'sanitize_text_field',
            'unify_banner_theme' => 'sanitize_text_field',
            'unify_banner_primary_color' => 'sanitize_hex_color',
            'unify_banner_text' => 'sanitize_textarea_field',
            'unify_banner_accept_text' => 'sanitize_text_field',
            'unify_banner_reject_text' => 'sanitize_text_field',
            'unify_banner_customize_text' => 'sanitize_text_field',
            'unify_privacy_page_id' => 'absint'
        ];
        
        foreach ($fields as $key => $sanitizer) {
            if (isset($_POST[$key])) {
                $value = isset($_POST[$key]) ? wp_unslash($_POST[$key]) : '';
                $value = call_user_func($sanitizer, $value);
                update_option($key, $value);
            }
        }
        
        // Handle checkbox fields (boolean options)
        $checkbox_fields = ['unify_script_blocking_enabled', 'unify_ccpa_dnsmi_enabled'];
        foreach ($checkbox_fields as $checkbox) {
            $value = isset($_POST[$checkbox]) ? true : false;
            update_option($checkbox, $value);
        }
        
        wp_send_json_success(['message' => 'Settings saved successfully.']);
    }

    public static function update_dsar_status() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }
        
        $id = absint($_POST['request_id']);
        $status = sanitize_text_field($_POST['status']);
        
        $allowed = ['pending', 'verified', 'processing', 'completed', 'rejected'];
        if (!in_array($status, $allowed)) {
             wp_send_json_error(['message' => 'Invalid status']);
             return;
        }
        
        $result = Unify_DSAR_Queue::update_status($id, $status);
        
        if ($result !== false) {
             wp_send_json_success(['message' => 'Request updated to ' . ucfirst($status)]);
        } else {
             wp_send_json_error(['message' => 'Database error']);
        }
    }

    public static function save_dsar_settings() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Unauthorized']);
        
        $fields = [
            'unify_dsar_admin_email' => 'sanitize_email',
            'unify_dsar_auto_sources' => [__CLASS__, 'sanitize_array_map'],
            'unify_dsar_auto_trigger_access' => 'absint'
        ];
        
        foreach ($fields as $key => $sanitizer) {
            // Premium check: Automation fields require premium
            if (strpos($key, 'unify_dsar_auto') !== false && !unify_is_premium()) {
                continue;
            }

            if (isset($_POST[$key])) {
                update_option($key, call_user_func($sanitizer, $_POST[$key]));
            } elseif ($sanitizer === [__CLASS__, 'sanitize_array_map']) {
                // Handle unchecked checkboxes (missing from POST)
                update_option($key, []);
            } elseif ($sanitizer === 'absint' && strpos($key, 'trigger') !== false) {
                 // Checkboxes: if missing, set to 0
                update_option($key, 0);
            }
        }
        wp_send_json_success(['message' => 'DSAR settings saved.']);
    }
    
    public static function sanitize_array_map($input) {
        if (!is_array($input)) return [];
        return array_map('sanitize_text_field', $input);
    }

    public static function save_ai_system() {
        check_ajax_referer('unify_save_ai_system', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $data = [
            'system_name' => sanitize_text_field($_POST['system_name']),
            'provider' => sanitize_text_field($_POST['provider']),
            'purpose' => sanitize_textarea_field($_POST['purpose']),
            'risk_level' => sanitize_text_field($_POST['risk_level']),
            'human_oversight' => isset($_POST['human_oversight']) ? 1 : 0
        ];
        
        $id = !empty($_POST['id']) ? absint($_POST['id']) : 0;
        
        if ($id > 0) {
            $result = Unify_AI_Registry::update_system($id, $data);
        } else {
            $result = Unify_AI_Registry::add_system($data);
        }
        
        if ($result !== false) {
            wp_send_json_success(['message' => 'Saved']);
        } else {
            wp_send_json_error('Database error or no changes made');
        }
    }

    public static function get_ai_system() {
        check_ajax_referer('unify_get_ai_system', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = absint($_POST['id']);
        $system = Unify_AI_Registry::get_system($id);
        
        if ($system) {
            wp_send_json_success($system);
        } else {
            wp_send_json_error('System not found');
        }
    }

    public static function delete_ai_system() {
        check_ajax_referer('unify_delete_ai_system', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = absint($_POST['id']);
        if (Unify_AI_Registry::delete_system($id)) {
            wp_send_json_success();
        } else {
            wp_send_json_error('Failed to delete');
        }
    }

    public static function run_ai_scan() {
        check_ajax_referer('unify_run_ai_scan', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $results = Unify_AI_Detector::scan();
        wp_send_json_success($results);
    }

    public static function preview_transparency_card() {
        check_ajax_referer('unify_preview_transparency_card', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = absint($_POST['id']);
        $card_html = Unify_Transparency_Card::generate_card($id);
        
        wp_send_json_success($card_html);
    }
    
    public static function save_nis2_settings() {
        check_ajax_referer('unify_save_nis2_settings', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        // Handle retention (single value)
        if (isset($_POST['unify_nis2_log_retention'])) {
            update_option('unify_nis2_log_retention', absint($_POST['unify_nis2_log_retention']));
        }
        
        // Handle checkbox groups (log events)
        $log_events = isset($_POST['unify_nis2_log_events']) && is_array($_POST['unify_nis2_log_events']) 
            ? array_map('sanitize_text_field', $_POST['unify_nis2_log_events']) 
            : [];
        update_option('unify_nis2_log_events', $log_events);
        
        // Handle individual checkboxes (boolean options)
        $bool_options = ['unify_nis2_anonymize_ip', 'unify_nis2_exclude_admins'];
        foreach ($bool_options as $opt) {
            $val = isset($_POST[$opt]) ? 1 : 0;
            update_option($opt, $val);
        }
        
        wp_send_json_success(['message' => 'NIS2 settings saved.']);
    }

    public static function export_nis2_logs() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        $filters = [
            'search' => isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '',
            'event_type' => isset($_GET['event_type']) ? sanitize_text_field($_GET['event_type']) : '',
            'severity' => isset($_GET['severity']) ? sanitize_text_field($_GET['severity']) : '',
            'date_start' => isset($_GET['date_start']) ? sanitize_text_field($_GET['date_start']) : '',
            'date_end' => isset($_GET['date_end']) ? sanitize_text_field($_GET['date_end']) : ''
        ];
        
        // Fetch all matching logs (high limit)
        $logs = Unify_Security_Logger::get_logs(5000, 0, $filters);
        
        // Log this export action (NIS2 Requirement: Audit the auditors)
        Unify_Security_Logger::log('data_export', 'Exported Activity Logs as CSV', 'info');
        
        // Clear any previous output
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        // Generate CSV content as a string
        $headers = ['Timestamp', 'Event Type', 'User ID', 'User Login', 'IP Address', 'Severity', 'Details'];
        $csv_content = '"' . implode('","', $headers) . '"' . "\n";
        
        foreach ($logs as $log) {
            $user_info = get_userdata($log->user_id);
            $login = $user_info ? $user_info->user_login : 'System';
            
            $row = [
                $log->created_at,
                $log->event_type,
                $log->user_id,
                $login,
                $log->ip_address,
                $log->severity,
                str_replace('"', '""', $log->details) // Escape double quotes for CSV
            ];
            $csv_content .= '"' . implode('","', $row) . '"' . "\n";
        }
        
        // Output CSV
        ignore_user_abort(true);
        if (ob_get_level()) ob_end_clean();
        
        // Set proper headers for download
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="nis2-activity-log-' . current_time('Y-m-d') . '.csv"');
        header('Content-Length: ' . strlen($csv_content));
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV content is properly escaped above
        echo $csv_content;
        exit;
    }
    
    public static function save_policy() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = !empty($_POST['id']) ? absint($_POST['id']) : 0;
        
        $data = [
            'title' => sanitize_text_field($_POST['title']),
            'content' => wp_kses_post($_POST['content']), // Allow HTML in content
            'version' => sanitize_text_field($_POST['version']),
            'status' => sanitize_text_field($_POST['status'])
        ];
        
        if ($id > 0) {
            $result = Unify_Policy_Manager::update_policy($id, $data);
            $message = 'Policy updated successfully.';
        } else {
            $result = Unify_Policy_Manager::create_policy($data);
            $message = 'Policy created successfully.';
        }
        
        if ($result !== false) {
            wp_send_json_success(['message' => $message]);
        } else {
            wp_send_json_error(['message' => 'Database error']);
        }
    }
    
    public static function get_policy() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = absint($_POST['id']);
        $policy = Unify_Policy_Manager::get_policy($id);
        
        if ($policy) {
            wp_send_json_success($policy);
        } else {
            wp_send_json_error('Policy not found');
        }
    }
    
    public static function delete_policy() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = absint($_POST['id']);
        
        if (Unify_Policy_Manager::delete_policy($id)) {
            wp_send_json_success(['message' => 'Policy deleted.']);
        } else {
            wp_send_json_error(['message' => 'Failed to delete policy']);
        }
    }
    
    public static function run_vuln_scan() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        // Get scan options from request
        $options = [
            'check_core' => isset($_POST['check_core']) ? (bool) $_POST['check_core'] : true,
            'check_plugins' => isset($_POST['check_plugins']) ? (bool) $_POST['check_plugins'] : true,
            'check_themes' => isset($_POST['check_themes']) ? (bool) $_POST['check_themes'] : true,
            'check_heuristics' => isset($_POST['check_heuristics']) ? (bool) $_POST['check_heuristics'] : true,
        ];
        
        $results = Unify_Vulnerability_Scanner::run_scan($options);
        
        // Format results for display or just return raw
        wp_send_json_success([
            'message' => 'Scan completed successfully.',
            'timestamp' => current_time('mysql'),
            'results' => $results
        ]);
    }
    
    public static function save_manual_vulnerability() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = !empty($_POST['id']) ? absint($_POST['id']) : 0;
        
        $data = [
            'target_type' => sanitize_text_field($_POST['target_type']),
            'target_slug' => sanitize_text_field($_POST['target_slug']),
            'severity' => sanitize_text_field($_POST['severity']),
            'title' => sanitize_text_field($_POST['title']),
            'description' => wp_kses_post($_POST['description']),
            'affected_versions' => isset($_POST['affected_versions']) && is_array($_POST['affected_versions']) 
                ? array_map('sanitize_text_field', $_POST['affected_versions']) 
                : [],
            'fixed_version' => isset($_POST['fixed_version']) ? sanitize_text_field($_POST['fixed_version']) : '',
            'cve_id' => isset($_POST['cve_id']) ? sanitize_text_field($_POST['cve_id']) : '',
        ];
        
        if ($id > 0) {
            $result = Unify_Vulnerability_Scanner::update_manual_vulnerability($id, $data);
            $message = 'Vulnerability rule updated successfully.';
        } else {
            $result = Unify_Vulnerability_Scanner::add_manual_vulnerability($data);
            $message = 'Vulnerability rule created successfully.';
        }
        
        if ($result !== false) {
            wp_send_json_success(['message' => $message]);
        } else {
            wp_send_json_error(['message' => 'Database error']);
        }
    }
    
    public static function get_manual_vulnerabilities() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $rules = Unify_Vulnerability_Scanner::get_manual_vulnerability_rules();
        wp_send_json_success($rules);
    }
    
    public static function delete_manual_vulnerability() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $id = absint($_POST['id']);
        
        if (Unify_Vulnerability_Scanner::remove_manual_vulnerability($id)) {
            wp_send_json_success(['message' => 'Vulnerability rule deleted.']);
        } else {
            wp_send_json_error(['message' => 'Failed to delete vulnerability rule']);
        }
    }
    
    public static function export_vuln_report() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        $scan_data = Unify_Vulnerability_Scanner::get_last_scan();
        $vulnerabilities = $scan_data['results'];
        
        // Clear any previous output
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        // Generate CSV content as a string
        $headers = ['Type', 'Component', 'Severity', 'Issue', 'Current Version', 'Fixed Version', 'CVE', 'Recommendation'];
        $csv_content = '"' . implode('","', $headers) . '"' . "\n";
        
        foreach ($vulnerabilities as $vuln) {
            $row = [
                $vuln['type'],
                $vuln['name'],
                $vuln['severity'],
                $vuln['issue'],
                $vuln['current_version'] ?? 'N/A',
                $vuln['fixed_version'] ?? 'N/A',
                $vuln['cve'] ?? 'N/A',
                $vuln['recommendation'] ?? 'Update to latest version'
            ];
            $csv_content .= '"' . implode('","', $row) . '"' . "\n";
        }
        
        // Output CSV
        ignore_user_abort(true);
        if (ob_get_level()) ob_end_clean();
        
        // Set proper headers for download
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="vulnerability-report-' . current_time('Y-m-d') . '.csv"');
        header('Content-Length: ' . strlen($csv_content));
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV content is properly escaped above
        echo $csv_content;
        exit;
    }
    
    public static function save_modules() {
        check_ajax_referer('unify_save_modules', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized action']);
        }
        
        $modules = isset($_POST['modules']) ? $_POST['modules'] : [];
        if (!is_array($modules)) {
            $modules = [];
        }
        
        // Sanitize module slugs
        $sanitized_modules = array_map('sanitize_text_field', $modules);
        
        // Validate that each module can be activated
        $validated_modules = [];
        foreach ($sanitized_modules as $slug) {
            if (Unify_Module_Manager::can_activate($slug)) {
                $validated_modules[] = $slug;
            }
        }
        
        // Save to database
        Unify_Module_Manager::set_active_modules($validated_modules);
        
        wp_send_json_success([
            'message' => 'Modules updated successfully! Reloading page...',
            'active_modules' => $validated_modules
        ]);
    }
    
    /**
     * Get consent statistics for reports
     */
    public static function get_overview_stats() {
        check_ajax_referer('unify_reports_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // No premium check needed - this is for free tier
        
        $stats = Unify_Reports::get_overview_stats();
        wp_send_json_success($stats);
    }

    public static function get_consent_stats() {
        check_ajax_referer('unify_reports_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check premium status for advanced analytics
        if (!unify_is_premium()) {
            wp_send_json_error(['message' => 'Premium feature']);
        }
        
        $start_date = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : current_time('Y-m-d', strtotime('-30 days'));
        $end_date = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : current_time('Y-m-d');
        
        $stats = Unify_Reports::get_consent_stats($start_date, $end_date);
        wp_send_json_success($stats);
    }
    
    /**
     * Get DSAR statistics for reports
     */
    public static function get_dsar_stats() {
        check_ajax_referer('unify_reports_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check premium status
        if (!unify_is_premium()) {
            wp_send_json_error(['message' => 'Premium feature']);
        }
        
        $start_date = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : current_time('Y-m-d', strtotime('-30 days'));
        $end_date = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : current_time('Y-m-d');
        
        $stats = Unify_Reports::get_dsar_stats($start_date, $end_date);
        wp_send_json_success($stats);
    }
    
    /**
     * Get security statistics for reports
     */
    public static function get_security_stats() {
        check_ajax_referer('unify_reports_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check premium status for advanced analytics
        if (!unify_is_premium()) {
            wp_send_json_error(['message' => 'Premium feature']);
        }
        
        $start_date = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : current_time('Y-m-d', strtotime('-30 days'));
        $end_date = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : current_time('Y-m-d');
        
        $stats = Unify_Reports::get_security_stats($start_date, $end_date);
        wp_send_json_success($stats);
    }
    
    /**
     * Export report as PDF
     */
    public static function export_report_pdf() {
        check_ajax_referer('unify_reports_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check premium status
        if (!unify_is_premium()) {
            wp_send_json_error(['message' => 'Premium feature']);
        }
        
        $report_type = isset($_POST['report_type']) ? sanitize_text_field($_POST['report_type']) : 'consent';
        $start_date = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : current_time('Y-m-d', strtotime('-30 days'));
        $end_date = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : current_time('Y-m-d');
        
        $result = Unify_Reports::generate_pdf_report($report_type, $start_date, $end_date);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        } else {
            wp_send_json_success(['file_path' => $result]);
        }
    }
    
    /**
     * Toggle checklist item completion
     */
    public static function toggle_checklist_item() {
        check_ajax_referer('unify_dashboard_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $item_id = isset($_POST['item_id']) ? sanitize_text_field($_POST['item_id']) : '';
        $completed = isset($_POST['completed']) ? (bool) $_POST['completed'] : false;
        
        if (empty($item_id)) {
            wp_send_json_error(['message' => 'Invalid item ID']);
        }
        
        $user_id = get_current_user_id();
        $completed_items = get_user_meta($user_id, 'unify_checklist_completed', true) ?: [];
        
        if ($completed) {
            if (!in_array($item_id, $completed_items)) {
                $completed_items[] = $item_id;
            }
        } else {
            $completed_items = array_diff($completed_items, [$item_id]);
        }
        
        update_user_meta($user_id, 'unify_checklist_completed', $completed_items);
        
        // Calculate new progress
        $progress = Unify_Dashboard_Data::get_checklist_progress();
        
        wp_send_json_success([
            'message' => 'Checklist updated',
            'progress' => $progress
        ]);
    }

	/**
	 * Activate License
	 */
	public static function activate_license() {
		check_ajax_referer( 'unify_license_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		$license_key = sanitize_text_field( $_POST['license_key'] );
		
		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'License key is required.' ) );
		}

		$result = Unify_License::activate_key( $license_key );

		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			wp_send_json_error( $result );
		}
	}

	/**
	 * Deactivate License
	 */
	public static function deactivate_license() {
		check_ajax_referer( 'unify_license_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		$result = Unify_License::deactivate_key();

		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			// Even if API fails, we clear local data, so we can treat as success for UI
			wp_send_json_success( array( 'message' => 'License deactivated.' ) );
		}
	}

    public static function save_general_settings() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $remove = isset($_POST['unify_remove_on_uninstall']) ? 1 : 0;
        update_option('unify_remove_on_uninstall', $remove);

        wp_send_json_success(['message' => 'Settings saved successfully.']);
    }
}
