# WCAG 2.2 Accessibility Conformance Plan

## Unify Compliance - Premium Feature Implementation

### Executive Summary

This plan implements WCAG 2.2 Level AA conformance for the cookie consent banner, positioning Unify Compliance as one of the few truly accessible GDPR plugins on the market. This will be a **Premium-only feature** with clear visibility in the UI through badges, settings sections, and marketing copy.

---

## Phase 1: Cookie Banner HTML Structure

### File: `public/views/cookie-banner.php`

**Current Issues:**
- No ARIA landmarks or roles
- No semantic structure for screen readers
- Missing live region announcements

**Changes Required:**

```php
// Add to banner container
role="dialog"
aria-modal="true"
aria-labelledby="unify-banner-title"
aria-describedby="unify-banner-description"
tabindex="-1"

// Add to heading
id="unify-banner-title"

// Add to description paragraph
id="unify-banner-description"

// Add to each button
aria-label with descriptive text
```

**New Premium-only attributes:**
- `data-wcag-mode="true"` when premium accessibility is enabled
- High contrast class toggle support

---

## Phase 2: JavaScript Accessibility Enhancements

### File: `public/assets/js/banner.js`

**Current Issues:**
- No focus trap in modal
- No keyboard navigation (Escape to close)
- No focus restoration on close
- No screen reader announcements

**New Accessibility Module (Premium-gated):**

```javascript
const A11yEnhancements = {
    // Focus trap for modal
    trapFocus: function(container) { ... },
    
    // Return focus to trigger element
    restoreFocus: function() { ... },
    
    // Handle Escape key
    handleKeyboard: function(e) { ... },
    
    // Announce to screen readers
    announce: function(message, priority) { ... },
    
    // Initialize all a11y features
    init: function() { ... }
};
```

**Keyboard Support Matrix:**

| Key | Action |
|-----|--------|
| `Tab` | Navigate between interactive elements |
| `Shift+Tab` | Navigate backwards |
| `Escape` | Close modal/reject non-essential |
| `Enter/Space` | Activate focused button |

**Focus Management:**
1. On banner show → Focus first interactive element
2. On modal open → Trap focus within modal
3. On modal close → Return focus to "Customize" button
4. On banner dismiss → Return focus to page content

---

## Phase 3: CSS Accessibility Enhancements

### File: `public/assets/css/banner.css`

**Current Issues:**
- No visible focus indicators
- Touch targets may be too small (< 24x24px)
- Color contrast unverified
- No reduced motion support

**New Styles Required:**

```css
/* Focus indicators (WCAG 2.4.7) */
.unify-btn:focus-visible {
    outline: 3px solid #005fcc;
    outline-offset: 2px;
}

/* Touch target size (WCAG 2.5.8 - new in 2.2) */
.unify-btn {
    min-height: 44px;
    min-width: 44px;
    padding: 12px 20px;
}

/* High Contrast Theme */
.unify-theme-high-contrast {
    --unify-bg: #000000;
    --unify-text: #ffffff;
    --unify-link: #ffff00;
    --unify-focus: #00ffff;
    --unify-btn-bg: #ffffff;
    --unify-btn-text: #000000;
}

/* Reduced motion (WCAG 2.3.3) */
@media (prefers-reduced-motion: reduce) {
    .unify-btn, #unify-cookie-banner, #unify-customize-modal {
        transition: none;
        animation: none;
    }
}

/* Forced colors mode / Windows High Contrast */
@media (forced-colors: active) {
    .unify-btn {
        border: 2px solid currentColor;
    }
}
```

**Color Contrast Requirements (WCAG 1.4.3):**
- Normal text: 4.5:1 minimum
- Large text (18px+): 3:1 minimum
- UI components: 3:1 minimum

---

## Phase 4: Admin Settings UI

### File: `admin/views/gdpr/cookie-banner.php`

**New Section: "Accessibility (WCAG 2.2)"**

Location: After "Script Blocking" section, before the end of the form.

```php
<div style="border-top: 1px solid var(--unify-border); padding-top: 20px; margin-top: 20px;">
    <h4 style="margin-top: 0; display: flex; align-items: center; gap: 8px;">
        <span class="dashicons dashicons-universal-access" style="color: var(--unify-primary-color);"></span>
        Accessibility (WCAG 2.2)
        <span class="unify-badge unify-badge-premium">Premium</span>
    </h4>
    
    <?php if (unify_is_premium()) : ?>
        <!-- Accessibility settings controls -->
        <div class="unify-wcag-badge">
            <img src="wcag-badge.svg" alt="WCAG 2.2 AA Conformant">
            <span>Your banner meets WCAG 2.2 Level AA standards</span>
        </div>
        
        <!-- Theme selector with High Contrast option -->
        <!-- Reduced motion toggle -->
        <!-- Focus indicator style options -->
    <?php else : ?>
        <!-- Premium upsell block -->
    <?php endif; ?>
</div>
```

**Visual Badge Design:**
- WCAG 2.2 AA badge with checkmark
- Color: Accessible blue (#0066cc) with white text
- Displays when premium + accessibility enabled

---

## Phase 5: Premium Gating Implementation

### File: `includes/gdpr/class-unify-cookie-banner.php`

**New Method:**

```php
public static function is_wcag_mode_enabled() {
    return unify_is_premium() && 
           get_option('unify_wcag_mode_enabled', true);
}
```

**Conditional Loading:**

```php
public static function enqueue_assets() {
    // Always load base styles
    wp_enqueue_style('unify-banner', ...);
    
    // Premium: Load accessibility enhancements
    if (self::is_wcag_mode_enabled()) {
        wp_enqueue_style('unify-banner-a11y', 
            Unify_PLUGIN_URL . 'public/assets/css/banner-a11y.css', ...);
        wp_enqueue_script('unify-banner-a11y', 
            Unify_PLUGIN_URL . 'public/assets/js/banner-a11y.js', ...);
    }
}
```

---

## Phase 6: Feature Visibility & Marketing

### 6.1 Dashboard Badge

**File: `admin/views/dashboard.php`**

Add to GDPR module card when premium:

```php
<?php if (unify_is_premium()) : ?>
<div class="unify-feature-badge">
    <span class="dashicons dashicons-universal-access"></span>
    WCAG 2.2 AA
</div>
<?php endif; ?>
```

### 6.2 Module Selector Description

**File: `admin/views/module-selector.php`**

Update GDPR module description:

```
"Cookie consent, DSAR management, and WCAG 2.2 accessible banner (Premium)"
```

### 6.3 Cookie Banner Settings Header

**File: `admin/views/gdpr/cookie-banner.php`**

Update the subtitle (line 30):

```php
<p>Customize the appearance and behavior of your cookie consent banner. 
   <?php if (unify_is_premium()) : ?>
   <span class="unify-badge unify-badge-success" style="vertical-align: middle;">
       <span class="dashicons dashicons-universal-access" style="font-size: 12px;"></span>
       WCAG 2.2 AA Compliant
   </span>
   <?php endif; ?>
</p>
```

### 6.4 Premium Upsell Block

For free users, show in the accessibility section:

```php
<div class="unify-premium-upsell">
    <span class="dashicons dashicons-universal-access"></span>
    <h4>WCAG 2.2 Accessibility</h4>
    <p>Make your cookie banner accessible to all users with:</p>
    <ul>
        <li>Full keyboard navigation</li>
        <li>Screen reader optimized</li>
        <li>High contrast mode</li>
        <li>Touch-friendly targets</li>
        <li>Reduced motion support</li>
    </ul>
    <a href="..." class="unify-btn unify-btn-premium">Upgrade to Premium</a>
</div>
```

### 6.5 readme.txt Update

Add to GDPR section:

```
*   **WCAG 2.2 AA Accessible Banner (Premium)** - Industry-leading accessibility with keyboard navigation, screen reader support, high contrast mode, and touch-friendly design.
```

---

## Phase 7: Automated Accessibility Testing

### New File: `tests/test-accessibility.php`

**Test Cases:**

| Test | WCAG Criterion | Check |
|------|----------------|-------|
| Focus visible | 2.4.7 | All interactive elements have visible focus |
| Keyboard operable | 2.1.1 | All functions available via keyboard |
| No keyboard trap | 2.1.2 | Focus can be moved away from any component |
| Focus order | 2.4.3 | Focus order preserves meaning |
| Touch target | 2.5.8 | Targets are at least 24x24px |
| Color contrast | 1.4.3 | Text meets 4.5:1 ratio |
| Name, Role, Value | 4.1.2 | ARIA attributes present and valid |

**Implementation:**

```php
class Unify_A11y_Tests {
    
    public function test_banner_has_dialog_role() {
        $html = $this->get_banner_html();
        $this->assertContains('role="dialog"', $html);
    }
    
    public function test_banner_has_aria_labelledby() {
        $html = $this->get_banner_html();
        $this->assertContains('aria-labelledby=', $html);
    }
    
    public function test_buttons_have_accessible_names() {
        $html = $this->get_banner_html();
        // Check each button has text content or aria-label
    }
    
    // ... additional tests
}
```

### Frontend Testing (axe-core integration)

Add optional axe-core testing for development:

```javascript
// Development only - loaded when WP_DEBUG is true
if (typeof axe !== 'undefined') {
    axe.run('#unify-cookie-banner').then(results => {
        if (results.violations.length) {
            console.warn('Accessibility violations:', results.violations);
        }
    });
}
```

---

## Phase 8: Implementation Checklist

### Files to Create:
- [ ] `public/assets/css/banner-a11y.css` - Accessibility-specific styles
- [ ] `public/assets/js/banner-a11y.js` - Accessibility enhancements module
- [ ] `admin/assets/images/wcag-badge.svg` - WCAG compliance badge
- [ ] `tests/test-accessibility.php` - Automated a11y tests

### Files to Modify:
- [ ] `public/views/cookie-banner.php` - Add ARIA attributes
- [ ] `public/assets/js/banner.js` - Load a11y module conditionally
- [ ] `public/assets/css/banner.css` - Base focus styles (non-premium)
- [ ] `includes/gdpr/class-unify-cookie-banner.php` - Premium gating logic
- [ ] `admin/views/gdpr/cookie-banner.php` - Settings section & badges
- [ ] `admin/views/dashboard.php` - WCAG badge on GDPR card
- [ ] `admin/views/module-selector.php` - Updated description
- [ ] `readme.txt` - Feature documentation

### New Options:
- `unify_wcag_mode_enabled` (default: true for premium)
- `unify_banner_theme` - Add 'high-contrast' option
- `unify_reduced_motion` (default: 'auto' = respect prefers-reduced-motion)

---

## Estimated Implementation Order

1. **Core HTML/ARIA** (Phase 1) - 1 hour
2. **CSS Focus & Contrast** (Phase 3) - 1 hour  
3. **JavaScript A11y Module** (Phase 2) - 2 hours
4. **Premium Gating** (Phase 5) - 30 minutes
5. **Admin UI & Badges** (Phase 4 & 6) - 1.5 hours
6. **Testing Suite** (Phase 7) - 1 hour
7. **Documentation** (readme.txt) - 30 minutes

**Total Estimated Time: ~8 hours**

---

## Success Criteria

1. Banner passes axe-core automated tests with 0 violations
2. Full keyboard navigation works without mouse
3. Screen reader (NVDA/VoiceOver) can understand and operate banner
4. High contrast mode meets 7:1 contrast ratio
5. All touch targets ≥ 44x44px on mobile
6. Premium badge visible in admin UI
7. All automated tests pass
