<?php
/**
 * Dashboard Data Provider
 *
 * Centralizes all data retrieval for the admin dashboard,
 * aggregating information from consent logs, DSAR requests,
 * security logs, and module configurations.
 *
 * @package Unify_Compliance
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Unify_Dashboard_Data {

	/**
	 * Calculate overall compliance score (0-100)
	 *
	 * Based on:
	 * - Cookie banner enabled (20 points)
	 * - Privacy policy exists (20 points)
	 * - DSAR form configured (20 points)
	 * - Activity logging enabled (20 points)
	 * - All DSAR requests processed (20 points, scaled)
	 *
	 * @return int Compliance score (0-100)
	 */
	public static function get_compliance_score() {
		$score = 0;

		// Cookie banner (20 pts) - only if GDPR module is active
		if ( Unify_Module_Manager::is_module_active( 'gdpr' ) ) {
			$banner_enabled = get_option( 'unify_banner_enabled' );
			// If option is not set (false), default to enabled (1) for compliance
			// Only penalize if explicitly disabled (0)
			if ( $banner_enabled === false || $banner_enabled == 1 ) {
				$score += 20;
			}
		}

		// Privacy policy (20 pts)
		$privacy_page = get_option( 'wp_page_for_privacy_policy' );
		if ( $privacy_page && get_post_status( $privacy_page ) === 'publish' ) {
			$score += 20;
		}

		// DSAR form (20 pts) - check if shortcode exists on any page
		global $wpdb;
		$dsar_page = $wpdb->get_var( "SELECT ID FROM {$wpdb->posts} WHERE post_content LIKE '%[unify_dsar_form]%' AND post_status = 'publish'" );
		if ( $dsar_page ) {
			$score += 20;
		}

		// Activity logging (20 pts)
		if ( Unify_Module_Manager::is_module_active( 'nis2' ) ) {
			$score += 20;
		}

		// DSAR queue health (20 pts)
		if ( class_exists( 'Unify_DSAR_Queue' ) ) {
			$pending = Unify_DSAR_Queue::count_by_status_single( 'pending' );
			$total   = Unify_DSAR_Queue::count_all();
			if ( $total === 0 ) {
				$score += 20; // No requests = fully compliant
			} elseif ( $pending === 0 ) {
				$score += 20; // All processed
			} else {
				$ratio  = ( $total - $pending ) / $total;
				$score += floor( 20 * $ratio );
			}
		} else {
			$score += 20; // If DSAR not active, don't penalize
		}

		return min( 100, $score );
	}

	/**
	 * Get GDPR module status and statistics
	 *
	 * @return array GDPR status data
	 */
	public static function get_gdpr_status() {
		$banner_enabled = get_option( 'unify_banner_enabled', 1 );
		$privacy_page   = get_option( 'wp_page_for_privacy_policy' );
		$privacy_exists = $privacy_page && get_post_status( $privacy_page ) === 'publish';

		global $wpdb;
		$dsar_page = $wpdb->get_var( "SELECT ID FROM {$wpdb->posts} WHERE post_content LIKE '%[unify_dsar_form]%' AND post_status = 'publish'" );

		// Get consent stats for last 7 days
		$consent_stats = array();
		if ( class_exists( 'Unify_Reports' ) ) {
			$consent_stats = Unify_Reports::get_consent_stats(
                current_time( 'Y-m-d', strtotime( '-7 days' ) ),
                current_time( 'Y-m-d' )
			);
		}

		// Get pending DSAR count
		$pending_dsar = 0;
		if ( class_exists( 'Unify_DSAR_Queue' ) ) {
			$pending_dsar = Unify_DSAR_Queue::count_by_status_single( 'pending' );
		}

		// Format sparkline data - ensure last 7 days are represented
		$consent_sparkline = array();
		$dates = array();
		for ( $i = 6; $i >= 0; $i-- ) {
			$date = gmdate( 'Y-m-d', strtotime( "-$i days" ) );
			$dates[ $date ] = 0;
		}

		if ( ! empty( $consent_stats['daily_stats'] ) ) {
			foreach ( $consent_stats['daily_stats'] as $day ) {
				if ( isset( $dates[ $day['date'] ] ) ) {
					$dates[ $day['date'] ] = intval( $day['total_consents'] );
				}
			}
		}

		foreach ( $dates as $date => $count ) {
			$consent_sparkline[] = array(
				'date'  => $date,
				'count' => $count,
			);
		}

		return array(
			'banner_enabled'    => $banner_enabled,
			'privacy_exists'    => $privacy_exists,
			'dsar_configured'   => ! empty( $dsar_page ),
			'dsar_page_url'     => $dsar_page ? get_permalink( $dsar_page ) : null,
			'pending_dsar'      => $pending_dsar,
			'total_consents'    => isset( $consent_stats['total_consents'] ) ? $consent_stats['total_consents'] : 0,
			'acceptance_rate'   => isset( $consent_stats['acceptance_rate'] ) ? $consent_stats['acceptance_rate'] : 0,
			'consent_sparkline' => $consent_sparkline,
		);
	}

	/**
	 * Get NIS2 module status and security metrics
	 *
	 * @return array NIS2 status data
	 */
	public static function get_nis2_status() {
		$security_stats = array();
		if ( class_exists( 'Unify_Reports' ) ) {
			$security_stats = Unify_Reports::get_security_stats(
                current_time( 'Y-m-d', strtotime( '-7 days' ) ),
                current_time( 'Y-m-d' )
			);
		}

		$retention_days = get_option( 'unify_nis2_log_retention', 0 );

		// Format sparkline data - ensure last 7 days are represented
		$failed_login_sparkline = array();
		$dates = array();
		for ( $i = 6; $i >= 0; $i-- ) {
			$date = gmdate( 'Y-m-d', strtotime( "-$i days" ) );
			$dates[ $date ] = 0;
		}

		if ( ! empty( $security_stats['failed_logins_daily'] ) ) {
			foreach ( $security_stats['failed_logins_daily'] as $day ) {
				if ( isset( $dates[ $day['date'] ] ) ) {
					$dates[ $day['date'] ] = intval( $day['failed_attempts'] );
				}
			}
		}

		foreach ( $dates as $date => $count ) {
			$failed_login_sparkline[] = array(
				'date'  => $date,
				'count' => $count,
			);
		}

		return array(
			'logging_enabled'        => Unify_Module_Manager::is_module_active( 'nis2' ),
			'retention_configured'   => $retention_days > 0,
			'retention_days'         => $retention_days,
			'security_score'         => isset( $security_stats['security_score'] ) ? $security_stats['security_score'] : 100,
			'failed_logins'          => isset( $security_stats['failed_logins'] ) ? $security_stats['failed_logins'] : 0,
			'successful_logins'      => isset( $security_stats['successful_logins'] ) ? $security_stats['successful_logins'] : 0,
			'suspicious_ips'         => isset( $security_stats['suspicious_ips'] ) ? count( $security_stats['suspicious_ips'] ) : 0,
			'failed_login_sparkline' => $failed_login_sparkline,
		);
	}

	/**
	 * Get quick stats for KPI cards
	 *
	 * @return array Array of stat objects
	 */
	public static function get_quick_stats() {
		$gdpr = self::get_gdpr_status();
		$nis2 = self::get_nis2_status();

		return array(
			array(
				'label'    => 'Total Consents',
				'value'    => $gdpr['total_consents'],
				'icon'     => 'thumbs-up',
				'color'    => '#10b981',
				'subtitle' => 'Last 7 days',
			),
			array(
				'label'    => 'DSAR Pending',
				'value'    => $gdpr['pending_dsar'],
				'icon'     => 'email-alt',
				'color'    => $gdpr['pending_dsar'] > 0 ? '#f59e0b' : '#10b981',
				'subtitle' => $gdpr['pending_dsar'] > 0 ? 'Action required' : 'All clear',
			),
			array(
				'label'    => 'Failed Logins',
				'value'    => $nis2['failed_logins'],
				'icon'     => 'shield',
				'color'    => $nis2['failed_logins'] > 10 ? '#ef4444' : '#10b981',
				'subtitle' => 'Last 7 days',
			),
			array(
				'label'    => 'Security Score',
				'value'    => $nis2['security_score'],
				'icon'     => 'awards',
				'color'    => $nis2['security_score'] >= 80 ? '#10b981' : ( $nis2['security_score'] >= 50 ? '#f59e0b' : '#ef4444' ),
				'subtitle' => 'Based on activity',
			),
		);
	}

	/**
	 * Get recent activity from security logs
	 *
	 * @param int $limit Number of events to retrieve
	 * @return array Array of activity events
	 */
	public static function get_recent_activity( $limit = 5 ) {
		if ( ! class_exists( 'Unify_Security_Logger' ) ) {
			return array();
		}

		$logs = Unify_Security_Logger::get_logs( $limit, 0, array() );

		if ( empty( $logs ) ) {
			return array();
		}

		return array_map(
			function ( $log ) {
				$user = $log->user_id ? get_userdata( $log->user_id ) : null;
				return array(
					'time'     => human_time_diff( strtotime( $log->created_at ) ) . ' ago',
					'time_raw' => $log->created_at,
					'user'     => $user ? $user->display_name : '—',
					'event'    => ucfirst( str_replace( '_', ' ', $log->event_type ) ),
					'details'  => $log->details,
					'severity' => $log->severity,
					'ip'       => $log->ip_address,
				);
			},
			$logs
		);
	}

	/**
	 * Get checklist items with completion status
	 *
	 * @return array Array of checklist items
	 */
	public static function get_checklist_items() {
		$user_id   = get_current_user_id();
		$completed = get_user_meta( $user_id, 'unify_checklist_completed', true ) ?: array();

		// Check for DSAR form shortcode
		global $wpdb;
		$dsar_page = $wpdb->get_var( "SELECT ID FROM {$wpdb->posts} WHERE post_content LIKE '%[unify_dsar_form]%' AND post_status = 'publish'" );

		// Check for security logs
		$logs_exist = false;
		if ( class_exists( 'Unify_Security_Logger' ) ) {
			// Fast check if any logs exist
			$logs_exist = (bool) $wpdb->get_var( "SELECT id FROM {$wpdb->prefix}unify_security_logs LIMIT 1" );
		}

		$items = array();

		// GDPR items
		if ( Unify_Module_Manager::is_module_active( 'gdpr' ) ) {
			$banner_enabled = get_option( 'unify_banner_enabled' );
			$is_banner_ok   = ( $banner_enabled === false || $banner_enabled == 1 ); // Default to true

			$items[] = array(
				'id'        => 'banner_enabled',
				'label'     => 'Enable Cookie Banner',
				'completed' => $is_banner_ok || in_array( 'banner_enabled', $completed ),
				'link'      => admin_url( 'admin.php?page=unify-gdpr&tab=cookie-banner' ),
				'help'      => 'Display a GDPR-compliant cookie consent banner to visitors.',
			);
			$items[] = array(
				'id'        => 'privacy_policy',
				'label'     => 'Create Privacy Policy',
				'completed' => get_option( 'wp_page_for_privacy_policy' ) ? true : in_array( 'privacy_policy', $completed ),
				'link'      => admin_url( 'options-privacy.php' ),
				'help'      => 'Create and publish a privacy policy page.',
			);
			$items[] = array(
				'id'        => 'dsar_form',
				'label'     => 'Configure DSAR Form',
				'completed' => $dsar_page ? true : in_array( 'dsar_form', $completed ),
				'link'      => admin_url( 'admin.php?page=unify-gdpr&tab=dsar-settings' ),
				'help'      => 'Set up the data subject access request form on your site.',
			);
		}

		// NIS2 items
		if ( Unify_Module_Manager::is_module_active( 'nis2' ) ) {
			$items[] = array(
				'id'        => 'log_retention',
				'label'     => 'Configure Log Retention',
				'completed' => get_option( 'unify_nis2_log_retention', 0 ) > 0 || in_array( 'log_retention', $completed ),
				'link'      => admin_url( 'admin.php?page=unify-nis2&tab=settings' ),
				'help'      => 'Set how long security logs should be retained.',
			);
			$items[] = array(
				'id'        => 'review_events',
				'label'     => 'Review Logged Events',
				'completed' => $logs_exist || in_array( 'review_events', $completed ),
				'link'      => admin_url( 'admin.php?page=unify-nis2&tab=activity-log' ),
				'help'      => 'Review which security events are being tracked.',
			);
		}

		return $items;
	}

	/**
	 * Get checklist completion percentage
	 *
	 * @return int Percentage (0-100)
	 */
	public static function get_checklist_progress() {
		$items     = self::get_checklist_items();
		$total     = count( $items );
		$completed = 0;

		if ( $total === 0 ) {
			return 100;
		}

		foreach ( $items as $item ) {
			if ( $item['completed'] ) {
				$completed++;
			}
		}

		return round( ( $completed / $total ) * 100 );
	}
}
