<?php
// includes/class-unify-database.php

defined('ABSPATH') || exit;

class Unify_Database {
    
    public static function init() {
        register_activation_hook(Unify_PLUGIN_FILE, [__CLASS__, 'install']);
    }

    public static function install() {
        global $wpdb;
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Table 1: Consent Logs
        $table_consent = $wpdb->prefix . 'unify_consent_logs';
        $sql_consent = "CREATE TABLE $table_consent (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            user_ip_hash VARCHAR(64) NOT NULL,
            consent_id VARCHAR(36) NOT NULL,
            consent_status ENUM('accepted_all', 'rejected_all', 'custom') NOT NULL,
            consent_categories JSON NOT NULL,
            user_agent VARCHAR(255),
            page_url VARCHAR(512),
            consent_date DATETIME NOT NULL,
            consent_expiry DATETIME NOT NULL,
            INDEX idx_date (consent_date),
            INDEX idx_ip (user_ip_hash),
            INDEX idx_consent_id (consent_id)
        ) $charset_collate;";
        
        dbDelta($sql_consent);
        
        // Table 2: DSAR Requests
        $table_dsar = $wpdb->prefix . 'unify_dsar_requests';
        $sql_dsar = "CREATE TABLE $table_dsar (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            email VARCHAR(255) NOT NULL,
            request_type ENUM('access', 'delete', 'portability') NOT NULL,
            status ENUM('pending', 'verified', 'processing', 'completed', 'rejected') DEFAULT 'pending',
            message TEXT,
            verification_code VARCHAR(6),
            verification_expires DATETIME,
            admin_notes TEXT,
            data_export_path VARCHAR(512),
            submitted_date DATETIME NOT NULL,
            verified_date DATETIME,
            completed_date DATETIME,
            completed_by BIGINT UNSIGNED,
            INDEX idx_status (status),
            INDEX idx_email (email),
            INDEX idx_date (submitted_date)
        ) $charset_collate;";
        
        dbDelta($sql_dsar);
        
        // Table 3: Detected Cookies (Premium)
        $table_cookies = $wpdb->prefix . 'unify_cookies';
        $sql_cookies = "CREATE TABLE $table_cookies (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            cookie_name VARCHAR(255) NOT NULL,
            cookie_domain VARCHAR(255),
            category ENUM('necessary', 'functional', 'analytics', 'marketing', 'unknown') DEFAULT 'unknown',
            description TEXT,
            expiry VARCHAR(50),
            is_third_party TINYINT(1) DEFAULT 0,
            first_detected DATETIME NOT NULL,
            last_seen DATETIME NOT NULL,
            is_active TINYINT(1) DEFAULT 1,
            UNIQUE KEY unique_cookie (cookie_name, cookie_domain)
        ) $charset_collate;";
        
        dbDelta($sql_cookies);
        
        // Table 4: Security Logs (NIS2)
        $table_logs = $wpdb->prefix . 'unify_security_logs';
        $sql_logs = "CREATE TABLE $table_logs (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            event_type VARCHAR(50) NOT NULL,
            user_id BIGINT UNSIGNED,
            details TEXT,
            ip_address VARCHAR(45),
            severity ENUM('info', 'warning', 'critical') DEFAULT 'info',
            created_at DATETIME NOT NULL,
            INDEX idx_created (created_at),
            INDEX idx_event (event_type)
        ) $charset_collate;";
        dbDelta($sql_logs);
        
        // Table 5: Policies (NIS2)
        $table_policies = $wpdb->prefix . 'unify_policies';
        $sql_policies = "CREATE TABLE $table_policies (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255) NOT NULL,
            content LONGTEXT,
            version VARCHAR(20) DEFAULT '1.0',
            last_updated DATETIME NOT NULL,
            updated_by BIGINT UNSIGNED,
            status ENUM('draft', 'published') DEFAULT 'draft'
        ) $charset_collate;";
        dbDelta($sql_policies);

        // Table 6: Manual Vulnerabilities (NIS2)
        $table_manual_vulns = $wpdb->prefix . 'unify_manual_vulnerabilities';
        $sql_manual_vulns = "CREATE TABLE $table_manual_vulns (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            target_type VARCHAR(20) NOT NULL,
            target_slug VARCHAR(255) NOT NULL,
            severity VARCHAR(20) NOT NULL,
            title VARCHAR(255) NOT NULL,
            description TEXT,
            affected_versions TEXT,
            fixed_version VARCHAR(50),
            cve_id VARCHAR(50),
            created_at DATETIME NOT NULL,
            created_by BIGINT UNSIGNED,
            INDEX idx_target_type (target_type),
            INDEX idx_target_slug (target_slug)
        ) $charset_collate;";
        dbDelta($sql_manual_vulns);

        // Table 7: AI Systems (AI Act)
        $table_ai_systems = $wpdb->prefix . 'unify_ai_systems';
        $sql_ai_systems = "CREATE TABLE $table_ai_systems (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            system_name VARCHAR(255) NOT NULL,
            provider VARCHAR(255),
            purpose TEXT,
            risk_level ENUM('minimal', 'limited', 'high', 'unacceptable') DEFAULT 'minimal',
            data_types TEXT,
            decision_impact TEXT,
            human_oversight TINYINT(1) DEFAULT 0,
            disclosure_page_id BIGINT UNSIGNED,
            created_date DATETIME NOT NULL,
            last_review DATETIME,
            INDEX idx_risk (risk_level)
        ) $charset_collate;";
        dbDelta($sql_ai_systems);

        // Table 8: AI Disclosures (AI Act)
        $table_ai_disclosures = $wpdb->prefix . 'unify_ai_disclosures';
        $sql_ai_disclosures = "CREATE TABLE $table_ai_disclosures (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            ai_system_id BIGINT UNSIGNED NOT NULL,
            disclosure_type VARCHAR(50) DEFAULT 'card',
            content LONGTEXT,
            generated_date DATETIME NOT NULL,
            INDEX idx_system (ai_system_id)
        ) $charset_collate;";
        dbDelta($sql_ai_disclosures);
        
        update_option('unify_db_version', Unify_VERSION);
    }
    
    public static function get_table_name($table) {
        global $wpdb;
        return $wpdb->prefix . 'unify_' . $table;
    }
}
