<?php
defined( 'ABSPATH' ) || exit;

class Unify_Geolocation {

    /**
     * Get 2-letter Country Code (ISO 3166-1 alpha-2)
     * 
     * Priority:
     * 1. Simulation Cookie (for Admin testing)
     * 2. Cloudflare Header (HTTP_CF_IPCOUNTRY)
     * 3. Fallback: 'XX' (Unknown)
     * 
     * @return string
     */
    public static function get_country_code() {
        // 1. Simulation Check
        if ( current_user_can( 'manage_options' ) ) {
            if ( isset( $_COOKIE['unify_simulate_country'] ) ) {
                return strtoupper( sanitize_text_field( $_COOKIE['unify_simulate_country'] ) );
            }
        }

        // 2. Cloudflare Header
        if ( isset( $_SERVER['HTTP_CF_IPCOUNTRY'] ) ) {
            return strtoupper( sanitize_text_field( $_SERVER['HTTP_CF_IPCOUNTRY'] ) );
        }

        // 3. Fallback
        return 'XX';
    }

    /**
     * Check if country is in EU/EEA
     * 
     * @param string $country_code
     * @return bool
     */
    public static function is_eu( $country_code = null ) {
        if ( ! $country_code ) {
            $country_code = self::get_country_code();
        }

        $eu_countries = [
            'AT', 'BE', 'BG', 'HR', 'CY', 'CZ', 'DK', 'EE', 'FI', 'FR', 'DE', 'GR', 'HU', 'IE', 'IT', 'LV', 'LT', 'LU', 'MT', 'NL', 'PL', 'PT', 'RO', 'SK', 'SI', 'ES', 'SE', // EU 27
            'IS', 'LI', 'NO', // EEA
            'GB', 'CH' // Include UK/Swiss often for GDPR alignment
        ];

        return in_array( strtoupper( $country_code ), $eu_countries );
    }

    /**
     * Check if visitor is likely from California (CCPA)
     * 
     * Since Cloudflare header only gives 'US', we treat all US visitors as potential CCPA
     * unless we use a more granular IP API (which is out of scope for MVP).
     * 
     * @param string $country_code
     * @return bool
     */
    public static function is_ccpa_region( $country_code = null ) {
        if ( ! $country_code ) {
            $country_code = self::get_country_code();
        }
        return strtoupper( $country_code ) === 'US';
    }
}
