<?php
defined( 'ABSPATH' ) || exit;

class Unify_Jurisdiction_Manager {
    
    /**
     * Get all configured rules sorted by priority
     * 
     * @return array
     */
    public static function get_rules() {
        $rules = get_option( 'unify_jurisdiction_rules', [] );
        if ( ! is_array( $rules ) ) {
            $rules = self::get_default_rules();
            update_option( 'unify_jurisdiction_rules', $rules );
        }
        
        // Sort by priority (asc)
        usort( $rules, function( $a, $b ) {
            return (int) $a['priority'] - (int) $b['priority'];
        });
        
        return $rules;
    }

    /**
     * Get default initial rules
     */
    public static function get_default_rules() {
        return [
            [
                'id' => 'rule_eu',
                'priority' => 10,
                'region_type' => 'preset_eu',
                'countries' => [],
                'action' => 'gdpr'
            ],
            [
                'id' => 'rule_us',
                'priority' => 20,
                'region_type' => 'countries',
                'countries' => ['US'],
                'action' => 'ccpa' // Future: or 'notice'
            ],
            [
                'id' => 'default',
                'priority' => 99,
                'region_type' => 'global',
                'countries' => [],
                'action' => 'notice'
            ]
        ];
    }

    /**
     * Determine which rule applies to the current visitor
     * 
     * @return array|null The matching rule
     */
    public static function get_applicable_rule() {
        $country = Unify_Geolocation::get_country_code();
        $rules = self::get_rules();

        foreach ( $rules as $rule ) {
            if ( self::matches_region( $rule, $country ) ) {
                return $rule;
            }
        }
        
        // Fallback if no global rule found (shouldn't happen if default exists)
        return null;
    }

    /**
     * Check if a country matches a rule's region definition
     */
    private static function matches_region( $rule, $country_code ) {
        $type = $rule['region_type'];
        
        if ( $type === 'global' ) {
            return true;
        }
        
        if ( $type === 'preset_eu' ) {
            return Unify_Geolocation::is_eu( $country_code );
        }
        
        if ( $type === 'countries' && isset( $rule['countries'] ) && is_array( $rule['countries'] ) ) {
            return in_array( $country_code, $rule['countries'] );
        }
        
        return false;
    }

    /**
     * Save a rule (add or update)
     */
    public static function save_rule( $rule_data ) {
        $rules = self::get_rules();
        
        // Generate ID if new
        if ( empty( $rule_data['id'] ) ) {
            $rule_data['id'] = 'rule_' . uniqid();
        }
        
        // Update existing or add new
        $found = false;
        foreach ( $rules as $k => $r ) {
            if ( $r['id'] === $rule_data['id'] ) {
                $rules[$k] = array_merge( $r, $rule_data );
                $found = true;
                break;
            }
        }
        
        if ( ! $found ) {
            $rules[] = $rule_data;
        }
        
        update_option( 'unify_jurisdiction_rules', $rules );
        return true;
    }

    /**
     * Delete a rule
     */
    public static function delete_rule( $id ) {
        if ( $id === 'default' ) return false; // Protect default
        
        $rules = self::get_rules();
        $new_rules = [];
        
        foreach ( $rules as $r ) {
            if ( $r['id'] !== $id ) {
                $new_rules[] = $r;
            }
        }
        
        update_option( 'unify_jurisdiction_rules', $new_rules );
        return true;
    }
}
