<?php
/**
 * License Management
 *
 * Handles license key validation, activation, and deactivation
 * via Lemon Squeezy API.
 *
 * @package Unify_Compliance
 * @since 1.0.3
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Unify_License {

	/**
	 * Options for storing license data.
	 */
	const OPTION_KEY        = 'unify_license_key';
	const STATUS_OPTION     = 'unify_license_status';
	const INSTANCE_OPTION   = 'unify_license_instance_id';
	const LAST_CHECK_OPTION = 'unify_license_last_check';
	const ERROR_OPTION      = 'unify_license_error';

	/**
	 * Lemon Squeezy API Endpoints.
	 * Replace with your actual store/product details where necessary,
	 * but endpoints are standard.
	 */
	const API_URL = 'https://api.lemonsqueezy.com/v1/licenses';

	/**
	 * Validate a license key.
	 *
	 * @param string $license_key The license key to validate.
	 * @return array Response from API.
	 */
	public static function validate_key( $license_key = null ) {
		if ( ! $license_key ) {
			$license_key = get_option( self::OPTION_KEY );
		}

		if ( ! $license_key ) {
			return array(
				'valid' => false,
				'error' => 'No license key provided.',
			);
		}

		// Prepare request
		$response = wp_remote_post(
			self::API_URL . '/validate',
			array(
				'body' => array(
					'license_key' => $license_key,
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'valid' => false,
				'error' => $response->get_error_message(),
			);
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $body['valid'] ) && $body['valid'] === true ) {
			return array(
				'valid' => true,
				'data'  => $body,
			);
		}

		return array(
			'valid' => false,
			'error' => isset( $body['error'] ) ? $body['error'] : 'Invalid license key.',
		);
	}

	/**
	 * Activate a license key.
	 *
	 * @param string $license_key The license key to activate.
	 * @param string $instance_name Name to identify this activation (default: site URL).
	 * @return array result with success boolean and message.
	 */
	public static function activate_key( $license_key, $instance_name = null ) {
		if ( ! $instance_name ) {
			$instance_name = get_site_url();
		}

		// Prepare request
		$response = wp_remote_post(
			self::API_URL . '/activate',
			array(
				'body' => array(
					'license_key'   => $license_key,
					'instance_name' => $instance_name,
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
			);
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $body['activated'] ) && $body['activated'] === true ) {
			// Save license data
			update_option( self::OPTION_KEY, $license_key );
			update_option( self::STATUS_OPTION, 'active' );
			update_option( self::INSTANCE_OPTION, $body['instance']['id'] );
			update_option( self::LAST_CHECK_OPTION, time() );
			delete_option( self::ERROR_OPTION );

			return array(
				'success' => true,
				'message' => 'License activated successfully.',
			);
		}

		$error = isset( $body['error'] ) ? $body['error'] : 'Activation failed.';
		update_option( self::ERROR_OPTION, $error );

		return array(
			'success' => false,
			'message' => $error,
		);
	}

	/**
	 * Deactivate the current license.
	 *
	 * @return array Result with success boolean.
	 */
	public static function deactivate_key() {
		$license_key = get_option( self::OPTION_KEY );
		$instance_id = get_option( self::INSTANCE_OPTION );

		if ( ! $license_key || ! $instance_id ) {
			// Local cleanup if data missing
			self::clear_license_data();
			return array( 'success' => true );
		}

		// Prepare request
		$response = wp_remote_post(
			self::API_URL . '/deactivate',
			array(
				'body' => array(
					'license_key' => $license_key,
					'instance_id' => $instance_id,
				),
			)
		);

		// Always clear local data on user request, even if API fails (assume force deactivate)
		self::clear_license_data();

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false, // Only indicates API failure, local is cleared
				'message' => $response->get_error_message(),
			);
		}

		return array(
			'success' => true,
			'message' => 'License deactivated.',
		);
	}

	/**
	 * Clear all local license data.
	 */
	private static function clear_license_data() {
		delete_option( self::OPTION_KEY );
		delete_option( self::STATUS_OPTION );
		delete_option( self::INSTANCE_OPTION );
		delete_option( self::LAST_CHECK_OPTION );
		delete_option( self::ERROR_OPTION );
	}

	/**
	 * Check if license is currently active.
	 *
	 * @return bool
	 */
	public static function is_active() {
		return get_option( self::STATUS_OPTION ) === 'active';
	}

	/**
	 * Get current license status details.
	 *
	 * @return array
	 */
	public static function get_status() {
		return array(
			'active'     => self::is_active(),
			'key'        => get_option( self::OPTION_KEY ), // Consider masking this for UI
			'key_masked' => self::mask_key( get_option( self::OPTION_KEY ) ),
			'last_check' => get_option( self::LAST_CHECK_OPTION ),
			'error'      => get_option( self::ERROR_OPTION ),
		);
	}

	/**
	 * Helper to mask license key.
	 */
	private static function mask_key( $key ) {
		if ( ! $key || strlen( $key ) < 10 ) {
			return $key;
		}
		return substr( $key, 0, 4 ) . '...' . substr( $key, -4 );
	}

	/**
	 * Schedule the daily validation cron.
	 */
	public static function schedule_validation_cron() {
		if ( ! wp_next_scheduled( 'unify_daily_license_check' ) ) {
			wp_schedule_event( time(), 'daily', 'unify_daily_license_check' );
		}
	}

	/**
	 * Unschedule the daily validation cron.
	 */
	public static function unschedule_validation_cron() {
		$timestamp = wp_next_scheduled( 'unify_daily_license_check' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'unify_daily_license_check' );
		}
	}

	/**
	 * Run daily validation.
	 * Hooked to 'unify_daily_license_check'.
	 */
	public static function run_daily_validation() {
		$license_key = get_option( self::OPTION_KEY );
		if ( ! $license_key || get_option( self::STATUS_OPTION ) !== 'active' ) {
			return;
		}

		$result = self::validate_key( $license_key );

		if ( $result['valid'] ) {
			// License is still valid
			update_option( self::LAST_CHECK_OPTION, time() );
			delete_option( self::ERROR_OPTION );
		} else {
			// License is no longer valid (expired, revoked, etc.)
			update_option( self::STATUS_OPTION, 'inactive' );
			update_option( self::ERROR_OPTION, isset( $result['error'] ) ? $result['error'] : 'License validation failed.' );
			update_option( self::LAST_CHECK_OPTION, time() );
			
			// Optional: Log this event if the logger is available
			if ( class_exists( 'Unify_Security_Logger' ) ) {
				// Unify_Security_Logger::log('system', 'license_validation_failed', 'high', $result['error']);
			}
		}
	}
}
