<?php
/**
 * Class Unify_Module_Manager
 * 
 * Manages the activation and deactivation of compliance modules.
 * Handles state persistence in wp_options.
 *
 * @package Unify_Compliance
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Unify_Module_Manager {

    /**
     * Option key for storing active modules.
     */
    const OPTION_KEY = 'unify_active_modules';

    /**
     * Get all available modules and their metadata.
     *
     * @return array
     */
    public static function get_all_modules() {
        return array(
            'gdpr' => array(
                'name'        => 'GDPR',
                'description' => 'General Data Protection Regulation. Includes Cookie Banner, DSAR forms, Consent Management, and WCAG 2.2 accessible banner (Premium).',
                'tier'        => 'free',
                'default'     => true,
                'icon'        => 'dashicons-shield-alt',
            ),
            'nis2' => array(
                'name'        => 'NIS2',
                'description' => 'Network and Information Security Directive. Includes Activity Logger, Vulnerability Scanning, and Security features.',
                'tier'        => 'free',
                'default'     => true,
                'icon'        => 'dashicons-lock',
            ),
            'ai-act' => array(
                'name'        => 'EU AI Act',
                'description' => 'Transparency requirements for AI systems. Includes AI System Registry and Disclaimers.',
                'tier'        => 'premium',
                'default'     => false,
                'icon'        => 'dashicons-superhero',
            ),
            'ccpa' => array(
                'name'        => 'CCPA',
                'description' => 'California Consumer Privacy Act features. Includes Do Not Sell My Info & CPRA updates.',
                'tier'        => 'premium',
                'default'     => false,
                'icon'        => 'dashicons-location-alt',
            ),
            'lgpd' => array(
                'name'        => 'LGPD',
                'description' => 'Brazilian General Data Protection Law.',
                'tier'        => 'coming_soon',
                'default'     => false,
                'icon'        => 'dashicons-flag',
            ),
        );
    }

    /**
     * Get list of currently active module slugs.
     *
     * @return array
     */
    public static function get_active_modules() {
        $defaults = array();
        foreach ( self::get_all_modules() as $slug => $data ) {
            if ( ! empty( $data['default'] ) ) {
                $defaults[] = $slug;
            }
        }

        return get_option( self::OPTION_KEY, $defaults );
    }

    /**
     * Save the list of active modules.
     *
     * @param array $modules Array of module slugs to activate.
     * @return bool True if value was updated, false otherwise.
     */
    public static function set_active_modules( $modules ) {
        // Ensure we only save valid modules
        $valid_slugs = array_keys( self::get_all_modules() );
        $sanitized_modules = array_intersect( $modules, $valid_slugs );
        
        // Also ensure we don't save modules the user can't access (like premium ones if they aren't premium)
        // For now, we'll allow saving all valid slugs, but in a real scenario we'd check licenses here.
        // TODO: Integrate with license check: if ( ! unify_is_premium() && $tier === 'premium' ) continue;

        return update_option( self::OPTION_KEY, $sanitized_modules );
    }

    /**
     * Check if a specific module is active.
     *
     * @param string $slug Module slug.
     * @return bool
     */
    public static function is_module_active( $slug ) {
        $active_modules = self::get_active_modules();
        return in_array( $slug, $active_modules, true );
    }

    /**
     * Check if a module can be activated by the current system/user.
     * 
     * @param string $slug Module slug.
     * @return bool
     */
    public static function can_activate( $slug ) {
        $modules = self::get_all_modules();
        
        if ( ! isset( $modules[ $slug ] ) ) {
            return false;
        }

        $module = $modules[ $slug ];

        if ( $module['tier'] === 'coming_soon' ) {
            return false;
        }

        // Check for premium tier if applicable
        if ( $module['tier'] === 'premium' ) {
            // Function unify_is_premium() should be defined in core or specific helper
            if ( function_exists('unify_is_premium') && ! unify_is_premium() ) {
                return false; 
            }
        }

        return true;
    }
}
