<?php
/**
 * Reports Generator Class
 *
 * Aggregates data from consent logs, DSAR requests, and activity logs
 * to generate compliance reports and analytics.
 *
 * @package Unify_Compliance
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Unify_Reports {

	/**
	 * Get consent statistics for a date range
	 *
	 * @param string $start_date Start date (Y-m-d format)
	 * @param string $end_date End date (Y-m-d format)
	 * @return array Consent statistics including trends and category breakdown
	 */
	public static function get_consent_stats( $start_date, $end_date ) {
		global $wpdb;
		$table = $wpdb->prefix . 'unify_consent_logs';

		// Check if table exists
		if ( $wpdb->get_var( "SHOW TABLES LIKE '$table'" ) !== $table ) {
			return self::get_empty_consent_stats();
		}

		// Get daily consent counts for trend chart
		$daily_stats = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				DATE(consent_date) as date,
				COUNT(*) as total_consents,
				SUM(CASE WHEN consent_status = 'accepted_all' THEN 1 ELSE 0 END) as accept_all,
				SUM(CASE WHEN JSON_CONTAINS(consent_categories, '\"functional\"') THEN 1 ELSE 0 END) as functional,
				SUM(CASE WHEN JSON_CONTAINS(consent_categories, '\"analytics\"') THEN 1 ELSE 0 END) as analytics,
				SUM(CASE WHEN JSON_CONTAINS(consent_categories, '\"marketing\"') THEN 1 ELSE 0 END) as marketing
			FROM $table
			WHERE consent_date BETWEEN %s AND %s
			GROUP BY DATE(consent_date)
			ORDER BY date ASC",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get overall totals for the period
		$totals = $wpdb->get_row( $wpdb->prepare(
			"SELECT 
				COUNT(*) as total_consents,
				SUM(CASE WHEN consent_status = 'accepted_all' THEN 1 ELSE 0 END) as accept_all,
				SUM(CASE WHEN JSON_CONTAINS(consent_categories, '\"functional\"') THEN 1 ELSE 0 END) as functional,
				SUM(CASE WHEN JSON_CONTAINS(consent_categories, '\"analytics\"') THEN 1 ELSE 0 END) as analytics,
				SUM(CASE WHEN JSON_CONTAINS(consent_categories, '\"marketing\"') THEN 1 ELSE 0 END) as marketing
			FROM $table
			WHERE consent_date BETWEEN %s AND %s",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Calculate percentages
		$total = intval( $totals['total_consents'] );
		$percentages = array(
			'necessary'  => 100, // Always 100%
			'functional' => $total > 0 ? round( ( intval( $totals['functional'] ) / $total ) * 100, 1 ) : 0,
			'analytics'  => $total > 0 ? round( ( intval( $totals['analytics'] ) / $total ) * 100, 1 ) : 0,
			'marketing'  => $total > 0 ? round( ( intval( $totals['marketing'] ) / $total ) * 100, 1 ) : 0,
		);

		// Calculate acceptance rate (accept all vs customize)
		$acceptance_rate = $total > 0 ? round( ( intval( $totals['accept_all'] ) / $total ) * 100, 1 ) : 0;

		return array(
			'total_consents'   => $total,
			'acceptance_rate'  => $acceptance_rate,
			'daily_stats'      => $daily_stats,
			'category_percentages' => $percentages,
			'period' => array(
				'start' => $start_date,
				'end'   => $end_date,
			),
		);
	}

	/**
	 * Get DSAR request statistics
	 *
	 * @param string $start_date Start date (Y-m-d format)
	 * @param string $end_date End date (Y-m-d format)
	 * @return array DSAR statistics including type breakdown and response times
	 */
	public static function get_dsar_stats( $start_date, $end_date ) {
		global $wpdb;
		$table = $wpdb->prefix . 'unify_dsar_requests';

		// Check if table exists
		if ( $wpdb->get_var( "SHOW TABLES LIKE '$table'" ) !== $table ) {
			return self::get_empty_dsar_stats();
		}

		// Get requests by type and status
		$by_type_status = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				request_type,
				status,
				COUNT(*) as count
			FROM $table
			WHERE submitted_date BETWEEN %s AND %s
			GROUP BY request_type, status",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get total by type
		$by_type = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				request_type,
				COUNT(*) as count
			FROM $table
			WHERE submitted_date BETWEEN %s AND %s
			GROUP BY request_type",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get average response time for completed requests
		$avg_response = $wpdb->get_var( $wpdb->prepare(
			"SELECT 
				AVG(TIMESTAMPDIFF(DAY, submitted_date, completed_date)) as avg_days
			FROM $table
			WHERE status = 'completed'
			AND completed_date BETWEEN %s AND %s",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		) );

		// Get monthly trend
		$monthly_trend = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				DATE_FORMAT(submitted_date, '%%Y-%%m') as month,
				COUNT(*) as count
			FROM $table
			WHERE submitted_date BETWEEN %s AND %s
			GROUP BY DATE_FORMAT(submitted_date, '%%Y-%%m')
			ORDER BY month ASC",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get recent requests
		$recent_requests = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				id,
				email,
				request_type,
				status,
				submitted_date
			FROM $table
			WHERE submitted_date BETWEEN %s AND %s
			ORDER BY submitted_date DESC
			LIMIT 10",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Calculate totals
		$total_requests = array_sum( array_column( $by_type, 'count' ) );

		return array(
			'total_requests'    => $total_requests,
			'by_type'           => $by_type,
			'by_type_status'    => $by_type_status,
			'avg_response_days' => $avg_response ? round( floatval( $avg_response ), 1 ) : 0,
			'monthly_trend'     => $monthly_trend,
			'recent_requests'   => $recent_requests,
			'period' => array(
				'start' => $start_date,
				'end'   => $end_date,
			),
		);
	}

	/**
	 * Get NIS2 security metrics
	 *
	 * @param string $start_date Start date (Y-m-d format)
	 * @param string $end_date End date (Y-m-d format)
	 * @return array Security metrics including login attempts and plugin updates
	 */
	public static function get_security_stats( $start_date, $end_date ) {
		global $wpdb;
		$table = $wpdb->prefix . 'unify_security_logs';

		// Check if table exists
		if ( $wpdb->get_var( "SHOW TABLES LIKE '$table'" ) !== $table ) {
			return self::get_empty_security_stats();
		}

		// Get daily failed login counts
		$failed_logins_daily = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				DATE(created_at) as date,
				COUNT(*) as failed_attempts
			FROM $table
			WHERE event_type = 'login_failed'
			AND created_at BETWEEN %s AND %s
			GROUP BY DATE(created_at)
			ORDER BY date ASC",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get successful logins
		$successful_logins = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) 
			FROM $table
			WHERE event_type = 'user_login'
			AND created_at BETWEEN %s AND %s",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		) );

		// Get total failed logins
		$failed_logins = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) 
			FROM $table
			WHERE event_type = 'login_failed'
			AND created_at BETWEEN %s AND %s",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		) );

		// Get top suspicious IPs
		$suspicious_ips = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				ip_address,
				COUNT(*) as attempts,
				MAX(created_at) as last_seen
			FROM $table
			WHERE event_type = 'login_failed'
			AND created_at BETWEEN %s AND %s
			GROUP BY ip_address
			ORDER BY attempts DESC
			LIMIT 10",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get plugin activations/deactivations
		$plugin_updates = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				details,
				created_at,
				user_id
			FROM $table
			WHERE event_type IN ('plugin_activated', 'plugin_deactivated')
			AND created_at BETWEEN %s AND %s
			ORDER BY created_at DESC
			LIMIT 20",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Get user role changes
		$role_changes = $wpdb->get_results( $wpdb->prepare(
			"SELECT 
				details,
				created_at,
				user_id
			FROM $table
			WHERE event_type = 'user_role_change'
			AND created_at BETWEEN %s AND %s
			ORDER BY created_at DESC
			LIMIT 10",
			$start_date . ' 00:00:00',
			$end_date . ' 23:59:59'
		), ARRAY_A );

		// Calculate security score (0-100)
		$security_score = self::calculate_security_score( array(
			'failed_logins'     => intval( $failed_logins ),
			'successful_logins' => intval( $successful_logins ),
			'plugin_updates'    => count( $plugin_updates ),
			'suspicious_ips'    => count( $suspicious_ips ),
		) );

		return array(
			'successful_logins'   => intval( $successful_logins ),
			'failed_logins'       => intval( $failed_logins ),
			'failed_logins_daily' => $failed_logins_daily,
			'suspicious_ips'      => $suspicious_ips,
			'plugin_updates'      => $plugin_updates,
			'role_changes'        => $role_changes,
			'security_score'      => $security_score,
			'period' => array(
				'start' => $start_date,
				'end'   => $end_date,
			),
		);
	}

	/**
	 * Calculate security score based on metrics
	 *
	 * @param array $metrics Security metrics
	 * @return int Security score (0-100)
	 */
	private static function calculate_security_score( $metrics ) {
		$score = 100;

		// Deduct points for failed logins
		$total_logins = $metrics['successful_logins'] + $metrics['failed_logins'];
		if ( $total_logins > 0 ) {
			$failed_ratio = $metrics['failed_logins'] / $total_logins;
			$score -= min( 30, $failed_ratio * 100 ); // Max 30 points deduction
		}

		// Deduct points for suspicious IPs
		$score -= min( 20, $metrics['suspicious_ips'] * 2 ); // Max 20 points deduction

		// Bonus points for plugin updates (shows maintenance)
		$score += min( 10, $metrics['plugin_updates'] * 2 ); // Max 10 points bonus

		return max( 0, min( 100, round( $score ) ) );
	}

	/**
	 * Get empty consent stats structure
	 *
	 * @return array Empty stats array
	 */
	private static function get_empty_consent_stats() {
		return array(
			'total_consents'       => 0,
			'acceptance_rate'      => 0,
			'daily_stats'          => array(),
			'category_percentages' => array(
				'necessary'  => 100,
				'functional' => 0,
				'analytics'  => 0,
				'marketing'  => 0,
			),
			'period' => array(
				'start' => '',
				'end'   => '',
			),
		);
	}

	/**
	 * Get empty DSAR stats structure
	 *
	 * @return array Empty stats array
	 */
	private static function get_empty_dsar_stats() {
		return array(
			'total_requests'    => 0,
			'by_type'           => array(),
			'by_type_status'    => array(),
			'avg_response_days' => 0,
			'monthly_trend'     => array(),
			'recent_requests'   => array(),
			'period' => array(
				'start' => '',
				'end'   => '',
			),
		);
	}

	/**
	 * Get empty security stats structure
	 *
	 * @return array Empty stats array
	 */
	private static function get_empty_security_stats() {
		return array(
			'successful_logins'   => 0,
			'failed_logins'       => 0,
			'failed_logins_daily' => array(),
			'suspicious_ips'      => array(),
			'plugin_updates'      => array(),
			'role_changes'        => array(),
			'security_score'      => 100,
			'period' => array(
				'start' => '',
				'end'   => '',
			),
		);
	}

	/**
	 * Generate PDF report
	 *
	 * @param string $report_type Type of report (consent, security, dsar, combined)
	 * @param string $start_date Start date (Y-m-d format)
	 * @param string $end_date End date (Y-m-d format)
	 * @return string|WP_Error File path on success, WP_Error on failure
	 */
	public static function generate_pdf_report( $report_type, $start_date, $end_date ) {
		// TODO: Implement PDF generation using FPDF or TCPDF
		// For now, return a placeholder
		return new WP_Error( 'not_implemented', __( 'PDF generation not yet implemented', 'unify-compliance' ) );
	}

	/**
	 * Get overview statistics for free tier dashboard
	 *
	 * @return array Overview statistics
	 */
	public static function get_overview_stats() {
		// Use fixed 30 day window for overview
        $end_date = current_time('Y-m-d');
        $start_date = current_time('Y-m-d', strtotime('-30 days'));

		$consent_stats = self::get_consent_stats($start_date, $end_date);
		$security_stats = self::get_security_stats($start_date, $end_date);
		$dsar_stats = self::get_dsar_stats($start_date, $end_date);

		return array(
			'consent' => array(
				'total' => $consent_stats['total_consents'],
				'acceptance_rate' => $consent_stats['acceptance_rate'],
				'period' => 'Last 30 Days'
			),
			'security' => array(
				'score' => $security_stats['security_score'],
				'failed_logins' => $security_stats['failed_logins'],
				'period' => 'Last 30 Days'
			),
			'dsar' => array(
				'total' => $dsar_stats['total_requests'],
				'pending' => 0, // Simplified
				'period' => 'Last 30 Days'
			)
		);
	}
}
