<?php
// includes/gdpr/class-unify-cookie-banner.php

defined('ABSPATH') || exit;

class Unify_Cookie_Banner {
    
    public static function init() {
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
        add_action('wp_footer', [__CLASS__, 'render_banner']);
        add_action('wp_ajax_unify_save_consent', [__CLASS__, 'ajax_save_consent']);
        add_action('wp_ajax_nopriv_unify_save_consent', [__CLASS__, 'ajax_save_consent']);
    }

    /**
     * Check if WCAG 2.2 accessibility mode is enabled
     * This is a Premium-only feature
     * 
     * @return bool
     */
    public static function is_wcag_mode_enabled() {
        return unify_is_premium() && get_option('unify_wcag_mode_enabled', true);
    }

    public static function enqueue_assets() {
        // Always load base styles
        wp_enqueue_style('unify-banner', Unify_PLUGIN_URL . 'public/assets/css/banner.css', [], Unify_VERSION);
        wp_enqueue_script('unify-banner', Unify_PLUGIN_URL . 'public/assets/js/banner.js', [], Unify_VERSION, true);
        
        // Premium: Load WCAG 2.2 accessibility enhancements
        if (self::is_wcag_mode_enabled()) {
            wp_enqueue_style(
                'unify-banner-a11y', 
                Unify_PLUGIN_URL . 'public/assets/css/banner-a11y.css', 
                ['unify-banner'], 
                Unify_VERSION
            );
            wp_enqueue_script(
                'unify-banner-a11y', 
                Unify_PLUGIN_URL . 'public/assets/js/banner-a11y.js', 
                ['unify-banner'], 
                Unify_VERSION, 
                true
            );
        }
        
        $settings = [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('unify_consent_nonce'),
            'expiry' => 365,
            'categories' => ['necessary', 'functional', 'analytics', 'marketing'],
            'wcag_enabled' => self::is_wcag_mode_enabled(),
        ];
        wp_localize_script('unify-banner', 'uwpcBannerParams', $settings);
    }

    public static function render_banner() {
        // Jurisdiction Logic
        if (class_exists('Unify_Jurisdiction_Manager')) {
            $rule = Unify_Jurisdiction_Manager::get_applicable_rule();
            if ($rule && isset($rule['action'])) {
                if ($rule['action'] === 'none') {
                    return; // Do not show banner
                }
                // Future: We can switch banner type (GDPR vs CCPA) here
                // For MVP, both map to the standard banner but we could change text
                // $action_type = $rule['action']; 
            }
        }

        $banner_text = Unify_Settings::get_option('unify_banner_text');
        $position = Unify_Settings::get_option('unify_banner_position');
        $theme = Unify_Settings::get_option('unify_banner_theme');
        $primary_color = Unify_Settings::get_option('unify_banner_primary_color');
        $accept_text = Unify_Settings::get_option('unify_banner_accept_text') ?: 'Accept All';
        $reject_text = Unify_Settings::get_option('unify_banner_reject_text') ?: 'Reject All';
        $customize_text = Unify_Settings::get_option('unify_banner_customize_text') ?: 'Customize';
        
        include Unify_PLUGIN_DIR . 'public/views/cookie-banner.php';
    }
    
    public static function ajax_save_consent() {
        check_ajax_referer('unify_consent_nonce', 'nonce');
        
        $consent_data = isset($_POST['consent']) ? $_POST['consent'] : [];
        $ip_hash = hash('sha256', $_SERVER['REMOTE_ADDR']);
        
        global $wpdb;
        $table = $wpdb->prefix . 'unify_consent_logs';
        
        // Sanitize categories JSON
        if (is_array($consent_data)) {
            $categories_json = json_encode(array_map('sanitize_text_field', $consent_data));
        } else {
            $categories_json = json_encode([]);
        }

        $wpdb->insert($table, [
            'user_ip_hash' => $ip_hash,
            'consent_id' => wp_generate_uuid4(),
            'consent_status' => 'custom',
            'consent_categories' => $categories_json,
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '',
            'page_url' => wp_get_referer(),
            'consent_date' => current_time('mysql'),
            'consent_expiry' => current_time('Y-m-d H:i:s', strtotime('+1 year')),
        ]);
        
        wp_send_json_success();
    }
}
