<?php
/**
 * Script Blocker for GDPR Cookie Consent
 *
 * Blocks non-essential tracking scripts until user consent is obtained.
 * Implements prior blocking as required by GDPR.
 *
 * @package Unify_Compliance
 * @since 1.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Unify_Script_Blocker {

	/**
	 * Known tracking script patterns organized by category
	 *
	 * @var array
	 */
	private static $blocked_patterns = array(
		'analytics' => array(
			// Google Analytics & Tag Manager
			'google-analytics.com',
			'googletagmanager.com',
			'gtag/js',
			'analytics.js',
			'ga.js',
			// Matomo/Piwik
			'matomo',
			'piwik',
			// Hotjar
			'hotjar.com',
			'static.hotjar.com',
			// Heap
			'heap-analytics',
			'heapanalytics.com',
			// Mixpanel
			'mixpanel.com',
			// Amplitude
			'amplitude.com',
			// Plausible
			'plausible.io',
			// Fathom
			'usefathom.com',
			// Clicky
			'static.getclicky.com',
		),
		'marketing' => array(
			// Facebook/Meta Pixel
			'connect.facebook.net',
			'facebook.com/tr',
			'fbevents.js',
			'fbq(',
			// Twitter/X
			'static.ads-twitter.com',
			'platform.twitter.com/widgets',
			// LinkedIn
			'snap.licdn.com',
			'linkedin.com/px',
			// TikTok
			'analytics.tiktok.com',
			// Pinterest
			'pintrk',
			's.pinimg.com',
			// Snapchat
			'sc-static.net',
			// Google Ads
			'googleadservices.com',
			'googlesyndication.com',
			'doubleclick.net',
			// Bing Ads
			'bat.bing.com',
			// Taboola
			'cdn.taboola.com',
			// Outbrain
			'outbrain.com',
			// Criteo
			'criteo.com',
			'criteo.net',
		),
		'functional' => array(
			// Chat widgets (optional blocking)
			'crisp.chat',
			'intercom.io',
			'tawk.to',
			'livechatinc.com',
			'drift.com',
			// Video embeds with tracking
			'youtube.com/iframe_api',
			'player.vimeo.com',
		),
	);

	/**
	 * User's current consent status
	 *
	 * @var array|null
	 */
	private static $consent = null;

	/**
	 * Whether script blocking is enabled
	 *
	 * @var bool
	 */
	private static $blocking_enabled = true;

	/**
	 * Blocked scripts for later re-enabling
	 *
	 * @var array
	 */
	private static $blocked_scripts = array();

	/**
	 * Initialize the script blocker
	 */
	public static function init() {
		// Check if blocking is enabled in settings
		self::$blocking_enabled = get_option( 'unify_script_blocking_enabled', true );

		if ( ! self::$blocking_enabled ) {
			return;
		}

		// Don't block in admin
		if ( is_admin() ) {
			return;
		}

		// Load consent status from cookie
		self::load_consent();

		// Hook into script loading
		add_filter( 'script_loader_tag', array( __CLASS__, 'maybe_block_script' ), 999, 3 );

		// Start output buffering to catch inline scripts
		add_action( 'wp_head', array( __CLASS__, 'start_output_buffer' ), 1 );
		add_action( 'wp_footer', array( __CLASS__, 'end_output_buffer' ), 999 );

		// Add blocked scripts data to footer
		add_action( 'wp_footer', array( __CLASS__, 'output_blocked_scripts_data' ), 5 );
	}

	/**
	 * Load user's consent status from cookie
	 */
	private static function load_consent() {
		$cookie_name = 'unify_consent';

		if ( isset( $_COOKIE[ $cookie_name ] ) ) {
			$consent_json = sanitize_text_field( wp_unslash( $_COOKIE[ $cookie_name ] ) );
			$consent_data = json_decode( $consent_json, true );

			if ( is_array( $consent_data ) ) {
				self::$consent = array(
					'necessary'  => true, // Always true
					'functional' => ! empty( $consent_data['functional'] ),
					'analytics'  => ! empty( $consent_data['analytics'] ),
					'marketing'  => ! empty( $consent_data['marketing'] ),
				);
			}
		}

		// If no consent cookie, block everything except necessary
		if ( self::$consent === null ) {
			self::$consent = array(
				'necessary'  => true,
				'functional' => false,
				'analytics'  => false,
				'marketing'  => false,
			);
		}
	}

	/**
	 * Get current consent status
	 *
	 * @return array Consent status array
	 */
	public static function get_consent() {
		if ( self::$consent === null ) {
			self::load_consent();
		}
		return self::$consent;
	}

	/**
	 * Check if a specific category is consented
	 *
	 * @param string $category Category name (analytics, marketing, functional).
	 * @return bool Whether consent is given for this category
	 */
	public static function has_consent( $category ) {
		$consent = self::get_consent();
		return ! empty( $consent[ $category ] );
	}

	/**
	 * Check if any consent has been given (cookie exists)
	 *
	 * @return bool Whether user has interacted with consent banner
	 */
	public static function has_made_choice() {
		return isset( $_COOKIE['unify_consent'] );
	}

	/**
	 * Filter to potentially block enqueued scripts
	 *
	 * @param string $tag    The script tag HTML.
	 * @param string $handle The script handle.
	 * @param string $src    The script source URL.
	 * @return string Modified script tag
	 */
	public static function maybe_block_script( $tag, $handle, $src ) {
		// Determine which category this script belongs to
		$category = self::get_script_category( $src, $handle );

		// If no blocking category found, allow script
		if ( $category === 'necessary' || $category === null ) {
			return $tag;
		}

		// Check if user has consented to this category
		if ( self::has_consent( $category ) ) {
			return $tag;
		}

		// Block the script by changing type and storing original
		$blocked_tag = self::block_script_tag( $tag, $src, $category );

		// Track blocked script
		self::$blocked_scripts[] = array(
			'handle'   => $handle,
			'src'      => $src,
			'category' => $category,
		);

		return $blocked_tag;
	}

	/**
	 * Determine which consent category a script belongs to
	 *
	 * @param string $src    Script source URL.
	 * @param string $handle Script handle.
	 * @return string|null Category name or null if not found
	 */
	private static function get_script_category( $src, $handle ) {
		// Check custom patterns from settings
		$custom_patterns = get_option( 'unify_custom_script_patterns', array() );

		if ( ! empty( $custom_patterns ) && is_array( $custom_patterns ) ) {
			foreach ( $custom_patterns as $pattern_data ) {
				if ( ! empty( $pattern_data['pattern'] ) && ! empty( $pattern_data['category'] ) ) {
					if ( stripos( $src, $pattern_data['pattern'] ) !== false ) {
						return sanitize_key( $pattern_data['category'] );
					}
				}
			}
		}

		// Check built-in patterns
		foreach ( self::$blocked_patterns as $category => $patterns ) {
			foreach ( $patterns as $pattern ) {
				if ( stripos( $src, $pattern ) !== false || stripos( $handle, $pattern ) !== false ) {
					return $category;
				}
			}
		}

		return null;
	}

	/**
	 * Convert a script tag to blocked format
	 *
	 * @param string $tag      Original script tag.
	 * @param string $src      Script source URL.
	 * @param string $category Consent category.
	 * @return string Blocked script tag
	 */
	private static function block_script_tag( $tag, $src, $category ) {
		// Change type to text/plain to prevent execution
		if ( strpos( $tag, 'type=' ) !== false ) {
			$tag = preg_replace( '/type=["\'][^"\']*["\']/', 'type="text/plain"', $tag );
		} else {
			$tag = str_replace( '<script', '<script type="text/plain"', $tag );
		}

		// Add data attributes for re-enabling
		$tag = str_replace(
			'<script',
			'<script data-consent-category="' . esc_attr( $category ) . '" data-blocked="true"',
			$tag
		);

		// If it has a src, store it in data-src as well
		if ( ! empty( $src ) ) {
			$tag = str_replace(
				'src=',
				'data-src="' . esc_url( $src ) . '" data-original-src=',
				$tag
			);
		}

		return $tag;
	}

	/**
	 * Start output buffering to catch inline scripts
	 */
	public static function start_output_buffer() {
		ob_start( array( __CLASS__, 'process_output_buffer' ) );
	}

	/**
	 * End output buffering
	 */
	public static function end_output_buffer() {
		if ( ob_get_level() > 0 ) {
			ob_end_flush();
		}
	}

	/**
	 * Process the output buffer to block inline scripts
	 *
	 * @param string $buffer The output buffer content.
	 * @return string Modified buffer
	 */
	public static function process_output_buffer( $buffer ) {
		if ( empty( $buffer ) ) {
			return $buffer;
		}

		// Find inline scripts that contain tracking code
		$buffer = preg_replace_callback(
			'/<script\b(?![^>]*data-blocked)[^>]*>(.*?)<\/script>/is',
			array( __CLASS__, 'maybe_block_inline_script' ),
			$buffer
		);

		return $buffer;
	}

	/**
	 * Check and potentially block an inline script
	 *
	 * @param array $matches Regex matches.
	 * @return string Modified script tag or original
	 */
	public static function maybe_block_inline_script( $matches ) {
		$full_tag   = $matches[0];
		$content    = isset( $matches[1] ) ? $matches[1] : '';
		$tag_open   = substr( $full_tag, 0, strpos( $full_tag, '>' ) + 1 );

		// Skip if already blocked or marked as necessary
		if ( strpos( $full_tag, 'data-blocked' ) !== false ) {
			return $full_tag;
		}

		if ( strpos( $full_tag, 'data-consent-category="necessary"' ) !== false ) {
			return $full_tag;
		}

		// Determine category from content
		$category = self::get_inline_script_category( $content );

		if ( $category === null || $category === 'necessary' ) {
			return $full_tag;
		}

		// Check consent
		if ( self::has_consent( $category ) ) {
			return $full_tag;
		}

		// Block the inline script
		$blocked_tag = str_replace( '<script', '<script type="text/plain" data-consent-category="' . esc_attr( $category ) . '" data-blocked="true"', $full_tag );

		// If original had type attribute, need to replace it
		$blocked_tag = preg_replace( '/type=["\']text\/javascript["\']/', 'type="text/plain"', $blocked_tag );

		return $blocked_tag;
	}

	/**
	 * Determine category for inline script based on content
	 *
	 * @param string $content Script content.
	 * @return string|null Category or null
	 */
	private static function get_inline_script_category( $content ) {
		// Analytics patterns in inline scripts
		$analytics_patterns = array(
			'gtag(',
			'ga(',
			'_gaq.push',
			'GoogleAnalyticsObject',
			'dataLayer.push',
			'_paq.push', // Matomo
			'hj(',       // Hotjar
			'heap.track',
			'mixpanel',
			'amplitude',
		);

		foreach ( $analytics_patterns as $pattern ) {
			if ( stripos( $content, $pattern ) !== false ) {
				return 'analytics';
			}
		}

		// Marketing patterns
		$marketing_patterns = array(
			'fbq(',
			'fbevents',
			'twq(',
			'_linkedin_partner_id',
			'ttq.track',
			'pintrk(',
			'snaptr(',
			'gtag(\'event\'',
			'adsbygoogle',
		);

		foreach ( $marketing_patterns as $pattern ) {
			if ( stripos( $content, $pattern ) !== false ) {
				return 'marketing';
			}
		}

		return null;
	}

	/**
	 * Output blocked scripts data for JavaScript re-enabling
	 */
	public static function output_blocked_scripts_data() {
		$consent = self::get_consent();
		?>
		<script type="text/javascript" id="unify-consent-data">
			window.unifyConsentData = {
				hasConsent: <?php echo self::has_made_choice() ? 'true' : 'false'; ?>,
				categories: <?php echo wp_json_encode( $consent ); ?>,
				blockedCount: <?php echo count( self::$blocked_scripts ); ?>
			};
		</script>
		<?php
	}

	/**
	 * Get list of blocked patterns for admin display
	 *
	 * @return array Blocked patterns by category
	 */
	public static function get_blocked_patterns() {
		return self::$blocked_patterns;
	}

	/**
	 * Add a custom pattern to block
	 *
	 * @param string $pattern  Pattern to match in script URL/content.
	 * @param string $category Consent category.
	 */
	public static function add_custom_pattern( $pattern, $category ) {
		$custom_patterns   = get_option( 'unify_custom_script_patterns', array() );
		$custom_patterns[] = array(
			'pattern'  => sanitize_text_field( $pattern ),
			'category' => sanitize_key( $category ),
		);
		update_option( 'unify_custom_script_patterns', $custom_patterns );
	}
}
