<?php
// includes/nis2/class-unify-security-logger.php

defined('ABSPATH') || exit;

class Unify_Security_Logger {
    
    public static function init() {
        // User Auth
        add_action('wp_login', [__CLASS__, 'log_login'], 10, 2);
        add_action('wp_logout', [__CLASS__, 'log_logout']);
        add_action('wp_login_failed', [__CLASS__, 'log_login_failed']);
        
        // User Management
        add_action('set_user_role', [__CLASS__, 'log_role_change'], 10, 3);
        add_action('delete_user', [__CLASS__, 'log_user_delete'], 10, 2);
        
        // Plugins & Themes (using upgrader_process_complete for updates/installs)
        add_action('activated_plugin', [__CLASS__, 'log_plugin_activation']);
        add_action('deactivated_plugin', [__CLASS__, 'log_plugin_deactivation']);
        // Note: upgrader_process_complete is complex, keeping it simple for now as per plan
        
        // Settings / Options
        add_action('updated_option', [__CLASS__, 'log_option_update'], 10, 3);
    }
    
    public static function should_log($event_type, $user_id = 0) {
        // Check Admin Exclusion
        if (get_option('unify_nis2_exclude_admins', 0)) {
            $user = $user_id ? get_userdata($user_id) : wp_get_current_user();
            if ($user && in_array('administrator', (array) $user->roles)) {
                return false;
            }
        }

        // Check Event Type Selection
        $enabled_events = get_option('unify_nis2_log_events', ['login', 'failed_login', 'plugin_change', 'user_role_change']);
        
        // Map hook events to categories
        $category = 'misc';
        if (strpos($event_type, 'login') !== false || $event_type === 'logout') $category = 'login';
        if ($event_type === 'login_failed') $category = 'failed_login';
        if (strpos($event_type, 'plugin') !== false || strpos($event_type, 'theme') !== false || strpos($event_type, 'option') !== false) $category = 'plugin_change';
        if (strpos($event_type, 'role') !== false || strpos($event_type, 'user') !== false) $category = 'user_role_change';
        
        return in_array($category, $enabled_events);
    }
    
    public static function log($event, $details = '', $severity = 'info', $user_id = null) {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_security_logs';
        
        if (!$user_id) {
            $user_id = get_current_user_id();
        }

        if (!self::should_log($event, $user_id)) {
            return;
        }
        
        // Sanitize
        $event = sanitize_text_field($event);
        $details = sanitize_text_field($details);
        
        // Anonymize IP
        $ip = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
        if (get_option('unify_nis2_anonymize_ip', 0) && !empty($ip)) {
            $ip = preg_replace('/(\d+)\.(\d+)\.(\d+)\.(\d+)/', '$1.$2.$3.xxx', $ip);
        }
        
        $wpdb->insert($table, [
            'event_type' => $event,
            'user_id' => $user_id,
            'details' => $details,
            'ip_address' => $ip,
            'severity' => $severity,
            'created_at' => current_time('mysql')
        ]);
    }
    
    public static function log_login($user_login, $user) {
        self::log('user_login', "User $user_login logged in.", 'info', $user->ID);
    }
    
    public static function log_logout($user_id) {
        $user = get_userdata($user_id);
        $name = $user ? $user->user_login : 'Unknown';
        self::log('user_logout', "User $name logged out.", 'info', $user_id);
    }
    
    public static function log_login_failed($username) {
        self::log('login_failed', "Failed login attempt for $username", 'warning', 0);
    }
    
    public static function log_plugin_activation($plugin) {
        self::log('plugin_activated', "Plugin activated: $plugin", 'info');
    }
    
    public static function log_plugin_deactivation($plugin) {
        self::log('plugin_deactivated', "Plugin deactivated: $plugin", 'warning');
    }

    public static function log_role_change($user_id, $role, $old_roles) {
        $user = get_userdata($user_id);
        self::log('user_role_change', "Role changed for {$user->user_login}: " . implode(',', $old_roles) . " -> $role", 'warning');
    }

    public static function log_user_delete($id, $reassign) {
        $user = get_userdata($id);
        $name = $user ? $user->user_login : "ID $id";
        self::log('user_deleted', "User account deleted: $name", 'critical');
    }
    
    public static function log_option_update($option, $old_value, $value) {
        $ignored = ['cron', 'rewrite_rules', 'unify_last_scan_date', 'action', 'unify_db_version']; 
        if (in_array($option, $ignored) || strpos($option, '_transient') === 0 || strpos($option, '_site_transient') === 0) return;
        
        self::log('option_updated', "Option '$option' changed.", 'info');
    }
    
    public static function get_logs($limit = 50, $offset = 0, $filters = []) {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_security_logs';
        
        $where = "WHERE 1=1";
        $args = [];
        
        if (!empty($filters['event_type'])) {
            $where .= " AND event_type = %s";
            $args[] = $filters['event_type'];
        }
        
        if (!empty($filters['search'])) {
            $where .= " AND (details LIKE %s OR event_type LIKE %s)";
            $args[] = '%' . $wpdb->esc_like($filters['search']) . '%';
            $args[] = '%' . $wpdb->esc_like($filters['search']) . '%';
        }
        
        if (!empty($filters['date_start'])) {
            $where .= " AND created_at >= %s";
            $args[] = $filters['date_start'] . ' 00:00:00';
        }

        if (!empty($filters['date_end'])) {
            $where .= " AND created_at <= %s";
            $args[] = $filters['date_end'] . ' 23:59:59';
        }
        
        if (!empty($filters['severity'])) {
            $where .= " AND severity = %s";
            $args[] = $filters['severity'];
        }

        $sql = "SELECT * FROM $table $where ORDER BY created_at DESC LIMIT %d OFFSET %d";
        $args[] = $limit;
        $args[] = $offset;
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQL.NotPrepared -- Table name is safe, comes from $wpdb->prefix
        return $wpdb->get_results($wpdb->prepare($sql, $args));
    }

    public static function count_logs($filters = []) {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_security_logs';
        
        $where = "WHERE 1=1";
        $args = [];
        
        // Duplicate filter logic (ideal would be a query builder class)
        if (!empty($filters['event_type'])) {
            $where .= " AND event_type = %s";
            $args[] = $filters['event_type'];
        }
        if (!empty($filters['search'])) {
            $where .= " AND (details LIKE %s OR event_type LIKE %s)";
            $args[] = '%' . $wpdb->esc_like($filters['search']) . '%';
            $args[] = '%' . $wpdb->esc_like($filters['search']) . '%';
        }

        if (!empty($filters['date_start'])) {
            $where .= " AND created_at >= %s";
            $args[] = $filters['date_start'] . ' 00:00:00';
        }

        if (!empty($filters['date_end'])) {
            $where .= " AND created_at <= %s";
            $args[] = $filters['date_end'] . ' 23:59:59';
        }

        if (!empty($filters['severity'])) {
            $where .= " AND severity = %s";
            $args[] = $filters['severity'];
        }
        
        if (!empty($args)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, comes from $wpdb->prefix
            return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table $where", $args));
        } else {
            return $wpdb->get_var("SELECT COUNT(*) FROM $table");
        }
    }
}
