<?php
// includes/premium/class-unify-cookie-scanner.php

defined('ABSPATH') || exit;

class Unify_Cookie_Scanner {
    
    public static function init() {
        add_action('wp_ajax_unify_run_scan', [__CLASS__, 'ajax_run_scan']);
    }
    
    public static function get_cookies() {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_cookies';
        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") != $table) {
            return [];
        }
        return $wpdb->get_results("SELECT * FROM $table ORDER BY category, cookie_name");
    }
    
    public static function ajax_run_scan() {
        check_ajax_referer('unify_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Unauthorized']);
        
        if (!unify_is_premium()) {
            wp_send_json_error(['message' => 'Premium feature required.']);
        }
        
        $url = home_url();
        // Use a real user agent to get actual page content
        $args = [
            'timeout' => 20,
            'user-agent' => 'Mozilla/5.0 (compatible; Unify_Compliance_Scanner/1.0; +https://example.com)',
            'redirection' => 5,
            'cookies' => [] // Start with empty jar
        ];
        
        $response = wp_remote_get($url, $args);
        
        $found_cookies = [];
        $scanned_signatures = self::get_known_signatures();

        // 1. Always present WP Cookies (Heuristic)
        $found_cookies['wordpress_test_cookie'] = [
            'category' => 'necessary',
            'description' => 'Used to check if the user\'s browser supports cookies.',
            'expiry' => 'Session'
        ];
        
        $found_cookies['wp-settings-{user_id}'] = [
            'category' => 'functional',
            'description' => 'Persists user dashboard interface adjustments.',
            'expiry' => '1 year'
        ];
        
        if (!is_wp_error($response)) {
            // 2. Header Analysis (Set-Cookie)
            $headers = wp_remote_retrieve_headers($response);
            if (isset($headers['set-cookie'])) {
                $cookies_header = $headers['set-cookie'];
                // wp_remote_get might return array or string depending on multiple headers
                if (!is_array($cookies_header)) {
                    $cookies_header = [$cookies_header];
                }
                
                foreach ($cookies_header as $cookie_string) {
                    $parts = explode(';', $cookie_string);
                    $name_val = explode('=', $parts[0], 2);
                    if (count($name_val) === 2) {
                        $name = trim($name_val[0]);
                        $category = self::guess_category($name);
                        $found_cookies[$name] = [
                            'category' => $category,
                            'description' => 'Detected via HTTP Header on your site homepage.',
                            'expiry' => 'Session' // Default if not parsed
                        ];
                        
                        // Try to find expiry
                        foreach ($parts as $part) {
                            if (stripos(trim($part), 'Max-Age') === 0 || stripos(trim($part), 'Expires') === 0) {
                                $found_cookies[$name]['expiry'] = trim($part);
                            }
                        }
                    }
                }
            }
            
            // 3. Document Body Analysis for Scripts
            $body = wp_remote_retrieve_body($response);
            
            foreach ($scanned_signatures as $service => $data) {
                $detected = false;
                foreach ($data['signatures'] as $sig) {
                    if (strpos($body, $sig) !== false) {
                        $detected = true;
                        break;
                    }
                }
                
                if ($detected) {
                    foreach ($data['cookies'] as $cookie_info) {
                        // Avoid overwriting if header scan found it (which is more accurate for expiry)
                        if (!isset($found_cookies[$cookie_info['name']])) {
                            $found_cookies[$cookie_info['name']] = [
                                'category' => $cookie_info['category'],
                                'description' => $cookie_info['description'],
                                'expiry' => $cookie_info['expiry']
                            ];
                        }
                    }
                }
            }
        }
        
        // Save to DB
        global $wpdb;
        $table = $wpdb->prefix . 'unify_cookies';
        
        // Check if table exists before truncating
        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") == $table) {
            $wpdb->query("TRUNCATE TABLE $table");
            
            foreach ($found_cookies as $name => $cookie) {
                $wpdb->insert($table, [
                    'cookie_name' => $name,
                    'category' => $cookie['category'],
                    'description' => $cookie['description'],
                    'expiry' => $cookie['expiry'],
                    'first_detected' => current_time('mysql'),
                    'last_seen' => current_time('mysql'),
                    'is_active' => 1
                ]);
            }
        }
        
        update_option('unify_last_scan_date', current_time('mysql'));

        wp_send_json_success([
            'message' => 'Scan complete. Found ' . count($found_cookies) . ' unique cookies.',
            'count' => count($found_cookies)
        ]);
    }
    
    private static function get_known_signatures() {
        return [
            'Google Analytics' => [
                'signatures' => ['google-analytics.com', 'gtag', 'ga.js', 'analytics.js', 'UA-'],
                'cookies' => [
                    ['name' => '_ga', 'category' => 'analytics', 'description' => 'Used to distinguish users.', 'expiry' => '2 years'],
                    ['name' => '_gid', 'category' => 'analytics', 'description' => 'Used to distinguish users.', 'expiry' => '24 hours'],
                    ['name' => '_gat', 'category' => 'analytics', 'description' => 'Used to throttle request rate.', 'expiry' => '1 minute']
                ]
            ],
            'Google Ads' => [
                'signatures' => ['googlesyndication.com', 'doubleclick.net', 'googleadservices.com'],
                'cookies' => [
                    ['name' => 'IDE', 'category' => 'marketing', 'description' => 'Used by Google DoubleClick to register and report the user\'s actions.', 'expiry' => '1 year'],
                    ['name' => 'test_cookie', 'category' => 'marketing', 'description' => 'Used to check if the user\'s browser supports cookies.', 'expiry' => '15 minutes']
                ]
            ],
            'Facebook' => [
                'signatures' => ['connect.facebook.net', 'fbevents.js', 'fbq('],
                'cookies' => [
                    ['name' => '_fbp', 'category' => 'marketing', 'description' => 'Used by Facebook to deliver a series of advertisement products.', 'expiry' => '3 months'],
                    ['name' => 'fr', 'category' => 'marketing', 'description' => 'Used by Facebook to deliver a series of advertisement products.', 'expiry' => '3 months']
                ]
            ],
            'YouTube' => [
                'signatures' => ['youtube.com/iframe_api', 'youtube.com/embed'],
                'cookies' => [
                    ['name' => 'VISITOR_INFO1_LIVE', 'category' => 'marketing', 'description' => 'Tries to estimate the users\' bandwidth on pages with integrated YouTube videos.', 'expiry' => '179 days'],
                    ['name' => 'YSC', 'category' => 'functional', 'description' => 'Registers a unique ID to keep statistics of what videos from YouTube the user has seen.', 'expiry' => 'Session'],
                    ['name' => 'PREF', 'category' => 'functional', 'description' => 'Stores user preferences.', 'expiry' => '8 months']
                ]
            ],
            'Hotjar' => [
                'signatures' => ['static.hotjar.com', 'hj('],
                'cookies' => [
                    ['name' => '_hjIncludedInSample', 'category' => 'analytics', 'description' => 'Determines if the user\'s navigation should be recorded.', 'expiry' => 'Session'],
                    ['name' => '_hjid', 'category' => 'analytics', 'description' => 'Hotjar ID that helps to identify a new user from a returning one.', 'expiry' => '1 year']
                ]
            ],
            'LinkedIn' => [
                'signatures' => ['linkedin.com/insight', 'snap.licdn.com'],
                'cookies' => [
                    ['name' => 'bcookie', 'category' => 'marketing', 'description' => 'Used by LinkedIn to track the use of embedded services.', 'expiry' => '2 years'],
                    ['name' => 'li_sugr', 'category' => 'marketing', 'description' => 'Used to track user behavior.', 'expiry' => '90 days']
                ]
            ],
            'Twitter' => [
                'signatures' => ['platform.twitter.com'],
                'cookies' => [
                    ['name' => 'personalization_id', 'category' => 'marketing', 'description' => 'Twitter integration and personalized ads.', 'expiry' => '2 years']
                ]
            ]
        ];
    }
    
    private static function guess_category($name) {
        // Simple heuristic for header-detected cookies
        if (strpos($name, 'wp-') === 0 || strpos($name, 'wordpress') === 0 || $name === 'PHPSESSID') {
            return 'necessary';
        }
        if (strpos($name, '_ga') === 0 || strpos($name, '_gid') === 0) {
            return 'analytics';
        }
        // Default fallbacks
        return 'functional';
    }
}
