<?php
// premium/class-unify-vulnerability-scanner.php

defined('ABSPATH') || exit;

class Unify_Vulnerability_Scanner {
    
    private static $core_cves = [
        '6.4.0' => [
            'CVE-2023-6000' => 'Property Oriented Programming (POP) chain vulnerability',
        ],
        '6.3.0' => [
            'CVE-2023-38000' => 'Shortcode block XSS vulnerability',
        ],
        '6.2.0' => [
            'CVE-2023-2745' => 'Stored XSS vulnerability in post content',
        ],
        '6.1.0' => [
            'CVE-2023-22622' => 'Directory traversal vulnerability',
        ],
        '6.0.0' => [
            'CVE-2022-43497' => 'Stored XSS vulnerability via shortcodes',
        ],
    ];
    
    /**
     * Run comprehensive vulnerability scan
     * 
     * @param array $options Scan options (core, plugins, themes, heuristics)
     * @return array Vulnerabilities found
     */
    public static function run_scan($options = []) {
        $defaults = [
            'check_core' => true,
            'check_plugins' => true,
            'check_themes' => true,
            'check_heuristics' => true,
        ];
        
        $options = wp_parse_args($options, $defaults);
        $vulnerabilities = [];
        
        // Check WordPress Core
        if ($options['check_core']) {
            $core_vulns = self::check_wordpress_core();
            $vulnerabilities = array_merge($vulnerabilities, $core_vulns);
        }
        
        // Check Plugins
        if ($options['check_plugins']) {
            $plugin_vulns = self::check_plugin_updates();
            $vulnerabilities = array_merge($vulnerabilities, $plugin_vulns);
            
            if ($options['check_heuristics']) {
                $heuristic_vulns = self::analyze_plugin_health();
                $vulnerabilities = array_merge($vulnerabilities, $heuristic_vulns);
            }
        }
        
        // Check Themes
        if ($options['check_themes']) {
            $theme_vulns = self::check_theme_updates();
            $vulnerabilities = array_merge($vulnerabilities, $theme_vulns);
        }
        
        // Check Manual Vulnerabilities
        $manual_vulns = self::get_manual_vulnerabilities();
        $vulnerabilities = array_merge($vulnerabilities, $manual_vulns);
        
        // Store results
        set_transient('unify_last_scan_result', $vulnerabilities, DAY_IN_SECONDS);
        set_transient('unify_last_scan_time', current_time('mysql'), DAY_IN_SECONDS);
        
        Unify_Security_Logger::log('vuln_scan', 'Vulnerability scan completed. Found ' . count($vulnerabilities) . ' issues.', 'info');
        
        return $vulnerabilities;
    }
    
    /**
     * Check WordPress Core for vulnerabilities
     * 
     * @return array Core vulnerabilities
     */
    public static function check_wordpress_core() {
        global $wp_version;
        $vulnerabilities = [];
        
        // Check for known CVEs in current version
        foreach (self::$core_cves as $vulnerable_version => $cves) {
            if (version_compare($wp_version, $vulnerable_version, '<=')) {
                foreach ($cves as $cve_id => $description) {
                    $vulnerabilities[] = [
                        'type' => 'core',
                        'name' => 'WordPress Core',
                        'severity' => 'critical',
                        'issue' => $description,
                        'current_version' => $wp_version,
                        'fixed_version' => null,
                        'cve' => $cve_id,
                        'recommendation' => 'Update WordPress to the latest version immediately.',
                    ];
                }
            }
        }
        
        // Check if core is outdated (using WordPress.org API)
        $core_update = self::get_core_update_info();
        if ($core_update && version_compare($wp_version, $core_update['version'], '<')) {
            $vulnerabilities[] = [
                'type' => 'core',
                'name' => 'WordPress Core',
                'severity' => 'high',
                'issue' => 'Outdated WordPress version detected',
                'current_version' => $wp_version,
                'fixed_version' => $core_update['version'],
                'cve' => null,
                'recommendation' => 'Update to WordPress ' . $core_update['version'],
            ];
        }
        
        return $vulnerabilities;
    }
    
    /**
     * Check plugins for updates and vulnerabilities
     * 
     * @return array Plugin vulnerabilities
     */
    public static function check_plugin_updates() {
        $vulnerabilities = [];
        $plugins = get_plugins();
        
        foreach ($plugins as $plugin_file => $plugin_data) {
            $slug = dirname($plugin_file);
            if ($slug === '.') {
                $slug = basename($plugin_file, '.php');
            }
            
            // Get plugin info from WordPress.org
            $api_data = self::get_plugin_info($slug);
            
            if ($api_data && isset($api_data['version'])) {
                // Check if update is available
                if (version_compare($plugin_data['Version'], $api_data['version'], '<')) {
                    $vulnerabilities[] = [
                        'type' => 'plugin',
                        'name' => $plugin_data['Name'],
                        'severity' => 'medium',
                        'issue' => 'Outdated plugin version',
                        'current_version' => $plugin_data['Version'],
                        'fixed_version' => $api_data['version'],
                        'cve' => null,
                        'recommendation' => 'Update to version ' . $api_data['version'],
                    ];
                }
            }
        }
        
        return $vulnerabilities;
    }
    
    /**
     * Check themes for updates
     * 
     * @return array Theme vulnerabilities
     */
    public static function check_theme_updates() {
        $vulnerabilities = [];
        $themes = wp_get_themes();
        
        foreach ($themes as $slug => $theme) {
            // Get theme info from WordPress.org
            $api_data = self::get_theme_info($slug);
            
            if ($api_data && isset($api_data['version'])) {
                // Check if update is available
                if (version_compare($theme->get('Version'), $api_data['version'], '<')) {
                    $vulnerabilities[] = [
                        'type' => 'theme',
                        'name' => $theme->get('Name'),
                        'severity' => 'medium',
                        'issue' => 'Outdated theme version',
                        'current_version' => $theme->get('Version'),
                        'fixed_version' => $api_data['version'],
                        'cve' => null,
                        'recommendation' => 'Update to version ' . $api_data['version'],
                    ];
                }
            }
        }
        
        return $vulnerabilities;
    }
    
    /**
     * Analyze plugin health using heuristics
     * 
     * @return array Heuristic vulnerabilities
     */
    public static function analyze_plugin_health() {
        $vulnerabilities = [];
        $plugins = get_plugins();
        
        foreach ($plugins as $plugin_file => $plugin_data) {
            $slug = dirname($plugin_file);
            if ($slug === '.') {
                $slug = basename($plugin_file, '.php');
            }
            
            // Get plugin info from WordPress.org
            $api_data = self::get_plugin_info($slug);
            
            if (!$api_data) {
                // Plugin not in WordPress.org repository
                $vulnerabilities[] = [
                    'type' => 'plugin',
                    'name' => $plugin_data['Name'],
                    'severity' => 'info',
                    'issue' => 'Plugin not found in WordPress.org repository',
                    'current_version' => $plugin_data['Version'],
                    'fixed_version' => null,
                    'cve' => null,
                    'recommendation' => 'Verify plugin source and authenticity. Consider using plugins from official repository.',
                ];
                continue;
            }
            
            $risk_score = self::calculate_risk_score($plugin_data, $api_data);
            
            if ($risk_score >= 70) {
                $vulnerabilities[] = [
                    'type' => 'plugin',
                    'name' => $plugin_data['Name'],
                    'severity' => 'high',
                    'issue' => 'High-risk plugin detected (abandoned or low quality)',
                    'current_version' => $plugin_data['Version'],
                    'fixed_version' => null,
                    'cve' => null,
                    'recommendation' => 'Consider replacing this plugin. Risk score: ' . $risk_score . '/100',
                ];
            } elseif ($risk_score >= 40) {
                $vulnerabilities[] = [
                    'type' => 'plugin',
                    'name' => $plugin_data['Name'],
                    'severity' => 'low',
                    'issue' => 'Medium-risk plugin detected',
                    'current_version' => $plugin_data['Version'],
                    'fixed_version' => null,
                    'cve' => null,
                    'recommendation' => 'Monitor this plugin closely. Risk score: ' . $risk_score . '/100',
                ];
            }
        }
        
        return $vulnerabilities;
    }
    
    /**
     * Calculate risk score for a plugin
     * 
     * @param array $plugin_data Plugin data from get_plugins()
     * @param array $api_data API data from WordPress.org
     * @return int Risk score (0-100)
     */
    private static function calculate_risk_score($plugin_data, $api_data) {
        $score = 0;
        
        // Last updated > 2 years ago
        if (isset($api_data['last_updated'])) {
            $last_updated = strtotime($api_data['last_updated']);
            if ($last_updated < strtotime('-2 years')) {
                $score += 30;
            } elseif ($last_updated < strtotime('-1 year')) {
                $score += 15;
            }
        }
        
        // Active installations < 1000
        if (isset($api_data['active_installs'])) {
            if ($api_data['active_installs'] < 1000) {
                $score += 20;
            } elseif ($api_data['active_installs'] < 10000) {
                $score += 10;
            }
        }
        
        // Rating < 3.0 (WordPress uses 0-100 scale, so < 60)
        if (isset($api_data['rating'])) {
            if ($api_data['rating'] < 60) {
                $score += 15;
            } elseif ($api_data['rating'] < 70) {
                $score += 8;
            }
        }
        
        // Low number of ratings (< 10)
        if (isset($api_data['num_ratings'])) {
            if ($api_data['num_ratings'] < 10) {
                $score += 10;
            }
        }
        
        return $score;
    }
    
    /**
     * Get plugin info from WordPress.org API
     * 
     * @param string $slug Plugin slug
     * @return array|false Plugin info or false on failure
     */
    private static function get_plugin_info($slug) {
        $cache_key = 'unify_plugin_info_' . $slug;
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $url = 'https://api.wordpress.org/plugins/info/1.2/?action=plugin_information&slug=' . $slug;
        $response = wp_remote_get($url, ['timeout' => 10]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || isset($data['error'])) {
            set_transient($cache_key, false, 12 * HOUR_IN_SECONDS);
            return false;
        }
        
        // Cache for 12 hours
        set_transient($cache_key, $data, 12 * HOUR_IN_SECONDS);
        
        return $data;
    }
    
    /**
     * Get theme info from WordPress.org API
     * 
     * @param string $slug Theme slug
     * @return array|false Theme info or false on failure
     */
    private static function get_theme_info($slug) {
        $cache_key = 'unify_theme_info_' . $slug;
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $url = 'https://api.wordpress.org/themes/info/1.2/?action=theme_information&slug=' . $slug;
        $response = wp_remote_get($url, ['timeout' => 10]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || isset($data['error'])) {
            set_transient($cache_key, false, 12 * HOUR_IN_SECONDS);
            return false;
        }
        
        // Cache for 12 hours
        set_transient($cache_key, $data, 12 * HOUR_IN_SECONDS);
        
        return $data;
    }
    
    /**
     * Get WordPress core update info
     * 
     * @return array|false Core update info or false
     */
    private static function get_core_update_info() {
        $cache_key = 'unify_core_update_info';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $url = 'https://api.wordpress.org/core/version-check/1.7/';
        $response = wp_remote_get($url, ['timeout' => 10]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || !isset($data['offers']) || empty($data['offers'])) {
            return false;
        }
        
        // Get the latest version
        $latest = $data['offers'][0];
        
        // Cache for 12 hours
        set_transient($cache_key, $latest, 12 * HOUR_IN_SECONDS);
        
        return $latest;
    }
    
    /**
     * Get manual vulnerabilities from database
     * 
     * @return array Manual vulnerabilities
     */
    public static function get_manual_vulnerabilities() {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_manual_vulnerabilities';
        
        $results = $wpdb->get_results("SELECT * FROM $table ORDER BY severity DESC", ARRAY_A);
        
        if (!$results) {
            return [];
        }
        
        $vulnerabilities = [];
        $plugins = get_plugins();
        $themes = wp_get_themes();
        
        foreach ($results as $row) {
            $is_applicable = false;
            $current_version = null;
            $component_name = $row['target_slug'];
            
            // Check if the target is installed
            if ($row['target_type'] === 'plugin') {
                foreach ($plugins as $plugin_file => $plugin_data) {
                    $slug = dirname($plugin_file);
                    if ($slug === '.') {
                        $slug = basename($plugin_file, '.php');
                    }
                    
                    if ($slug === $row['target_slug']) {
                        $is_applicable = true;
                        $current_version = $plugin_data['Version'];
                        $component_name = $plugin_data['Name'];
                        
                        // Check if version matches affected versions
                        if ($row['affected_versions']) {
                            $affected = json_decode($row['affected_versions'], true);
                            if ($affected && !self::version_matches_pattern($current_version, $affected)) {
                                $is_applicable = false;
                            }
                        }
                        break;
                    }
                }
            } elseif ($row['target_type'] === 'theme') {
                if (isset($themes[$row['target_slug']])) {
                    $is_applicable = true;
                    $current_version = $themes[$row['target_slug']]->get('Version');
                    $component_name = $themes[$row['target_slug']]->get('Name');
                    
                    // Check if version matches affected versions
                    if ($row['affected_versions']) {
                        $affected = json_decode($row['affected_versions'], true);
                        if ($affected && !self::version_matches_pattern($current_version, $affected)) {
                            $is_applicable = false;
                        }
                    }
                }
            } elseif ($row['target_type'] === 'core') {
                global $wp_version;
                $is_applicable = true;
                $current_version = $wp_version;
                $component_name = 'WordPress Core';
                
                // Check if version matches affected versions
                if ($row['affected_versions']) {
                    $affected = json_decode($row['affected_versions'], true);
                    if ($affected && !self::version_matches_pattern($current_version, $affected)) {
                        $is_applicable = false;
                    }
                }
            }
            
            if ($is_applicable) {
                $vulnerabilities[] = [
                    'type' => $row['target_type'],
                    'name' => $component_name,
                    'severity' => $row['severity'],
                    'issue' => $row['title'],
                    'current_version' => $current_version,
                    'fixed_version' => $row['fixed_version'],
                    'cve' => $row['cve_id'],
                    'recommendation' => $row['description'],
                    'manual' => true,
                ];
            }
        }
        
        return $vulnerabilities;
    }
    
    /**
     * Check if version matches pattern
     * 
     * @param string $version Current version
     * @param array $patterns Array of version patterns
     * @return bool True if matches
     */
    private static function version_matches_pattern($version, $patterns) {
        foreach ($patterns as $pattern) {
            // Support wildcards like "1.2.*"
            $pattern = str_replace('*', '.*', $pattern);
            $pattern = '/^' . $pattern . '$/';
            
            if (preg_match($pattern, $version)) {
                return true;
            }
            
            // Support version ranges like "<= 1.2.3"
            if (preg_match('/^([<>=!]+)\s*(.+)$/', $pattern, $matches)) {
                $operator = $matches[1];
                $compare_version = $matches[2];
                
                if (version_compare($version, $compare_version, $operator)) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Add manual vulnerability
     * 
     * @param array $data Vulnerability data
     * @return int|false Insert ID or false on failure
     */
    public static function add_manual_vulnerability($data) {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_manual_vulnerabilities';
        
        $insert_data = [
            'target_type' => sanitize_text_field($data['target_type']),
            'target_slug' => sanitize_text_field($data['target_slug']),
            'severity' => sanitize_text_field($data['severity']),
            'title' => sanitize_text_field($data['title']),
            'description' => wp_kses_post($data['description']),
            'affected_versions' => isset($data['affected_versions']) ? json_encode($data['affected_versions']) : null,
            'fixed_version' => isset($data['fixed_version']) ? sanitize_text_field($data['fixed_version']) : null,
            'cve_id' => isset($data['cve_id']) ? sanitize_text_field($data['cve_id']) : null,
            'created_at' => current_time('mysql'),
            'created_by' => get_current_user_id(),
        ];
        
        $result = $wpdb->insert($table, $insert_data);
        
        if ($result) {
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    /**
     * Update manual vulnerability
     * 
     * @param int $id Vulnerability ID
     * @param array $data Vulnerability data
     * @return bool Success
     */
    public static function update_manual_vulnerability($id, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_manual_vulnerabilities';
        
        $update_data = [
            'target_type' => sanitize_text_field($data['target_type']),
            'target_slug' => sanitize_text_field($data['target_slug']),
            'severity' => sanitize_text_field($data['severity']),
            'title' => sanitize_text_field($data['title']),
            'description' => wp_kses_post($data['description']),
            'affected_versions' => isset($data['affected_versions']) ? json_encode($data['affected_versions']) : null,
            'fixed_version' => isset($data['fixed_version']) ? sanitize_text_field($data['fixed_version']) : null,
            'cve_id' => isset($data['cve_id']) ? sanitize_text_field($data['cve_id']) : null,
        ];
        
        return $wpdb->update($table, $update_data, ['id' => absint($id)]);
    }
    
    /**
     * Delete manual vulnerability
     * 
     * @param int $id Vulnerability ID
     * @return bool Success
     */
    public static function remove_manual_vulnerability($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_manual_vulnerabilities';
        
        return $wpdb->delete($table, ['id' => absint($id)]);
    }
    
    /**
     * Get all manual vulnerability rules (for admin UI)
     * 
     * @return array Manual vulnerability rules
     */
    public static function get_manual_vulnerability_rules() {
        global $wpdb;
        $table = $wpdb->prefix . 'unify_manual_vulnerabilities';
        
        return $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC", ARRAY_A);
    }
    
    /**
     * Get last scan results
     * 
     * @return array Scan results
     */
    public static function get_last_scan() {
        return [
            'results' => get_transient('unify_last_scan_result') ?: [],
            'time' => get_transient('unify_last_scan_time')
        ];
    }
    
    /**
     * Clear all caches
     */
    public static function clear_cache() {
        delete_transient('unify_last_scan_result');
        delete_transient('unify_last_scan_time');
        delete_transient('unify_core_update_info');
        
        // Clear plugin/theme info caches
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_unify_plugin_info_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_unify_theme_info_%'");
    }
}
