/**
 * WCAG 2.2 AA Accessibility Enhancements (Premium Only)
 * Provides keyboard navigation, focus management, and screen reader support
 * @package Unify_Compliance
 */

(function () {
    'use strict';

    // Check if WCAG mode is enabled
    const isWCAGMode = document.getElementById('unify-cookie-banner')?.dataset.wcagMode === 'true';

    if (!isWCAGMode) {
        // WCAG features are premium-only
        return;
    }

    /**
     * Accessibility Enhancements Module
     */
    const A11yEnhancements = {
        focusableElements: 'button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])',
        previousFocus: null,
        announcer: null,
        currentModal: null,
        currentModalOverlay: null,

        /**
         * Initialize accessibility features
         */
        init: function () {
            this.createAnnouncer();
            this.setupBannerFocus();
            this.setupKeyboardHandlers();
        },

        /**
         * Create screen reader announcer element
         */
        createAnnouncer: function () {
            if (!this.announcer) {
                this.announcer = document.createElement('div');
                this.announcer.className = 'unify-a11y-announcer';
                this.announcer.setAttribute('role', 'status');
                this.announcer.setAttribute('aria-live', 'polite');
                this.announcer.setAttribute('aria-atomic', 'true');
                document.body.appendChild(this.announcer);
            }
        },

        /**
         * Announce message to screen readers
         * @param {string} message - Message to announce
         * @param {string} priority - 'polite' or 'assertive'
         */
        announce: function (message, priority = 'polite') {
            if (!this.announcer) return;

            this.announcer.setAttribute('aria-live', priority);
            // Clear and set message
            this.announcer.textContent = '';
            setTimeout(() => {
                this.announcer.textContent = message;
            }, 100);
        },

        /**
         * Setup focus management for banner
         */
        setupBannerFocus: function () {
            const banner = document.getElementById('unify-cookie-banner');
            if (!banner) return;

            // Store original display value
            const originalDisplay = banner.style.display;

            // Watch for banner visibility changes
            const observer = new MutationObserver((mutations) => {
                mutations.forEach((mutation) => {
                    if (mutation.type === 'attributes' && mutation.attributeName === 'style') {
                        const isVisible = banner.style.display !== 'none' &&
                            banner.style.display !== '' ||
                            window.getComputedStyle(banner).display !== 'none';

                        if (isVisible && mutation.oldValue?.includes('display: none')) {
                            this.onBannerShow(banner);
                        }
                    }
                });
            });

            observer.observe(banner, {
                attributes: true,
                attributeOldValue: true,
                attributeFilter: ['style']
            });
        },

        /**
         * Handle banner show event
         * @param {HTMLElement} banner
         */
        onBannerShow: function (banner) {
            // Store current focus
            this.previousFocus = document.activeElement;

            // Focus the banner
            setTimeout(() => {
                banner.focus();
                this.announce('Cookie consent dialog opened. Use tab to navigate options, escape to close.');
            }, 100);
        },

        /**
         * Setup keyboard event handlers
         */
        setupKeyboardHandlers: function () {
            document.addEventListener('keydown', (e) => {
                const banner = document.getElementById('unify-cookie-banner');
                const modal = document.getElementById('unify-customize-modal');

                // Handle Escape key
                if (e.key === 'Escape' || e.keyCode === 27) {
                    if (modal && modal.style.display !== 'none') {
                        this.closeModal(modal);
                        e.preventDefault();
                    } else if (banner && banner.style.display !== 'none') {
                        this.rejectAndClose();
                        e.preventDefault();
                    }
                }

                // Tab trap in modal - use currentModal (the dialog content) for proper focus containment
                if ((e.key === 'Tab' || e.keyCode === 9) && this.currentModal) {
                    this.trapFocus(this.currentModal, e);
                }
            });
        },

        /**
         * Trap focus within container
         * @param {HTMLElement} container
         * @param {KeyboardEvent} e
         */
        trapFocus: function (container, e) {
            const focusable = Array.from(container.querySelectorAll(this.focusableElements))
                .filter(el => el.offsetParent !== null); // Only visible elements

            if (focusable.length === 0) return;

            const firstFocusable = focusable[0];
            const lastFocusable = focusable[focusable.length - 1];

            if (e.shiftKey) {
                // Shift + Tab
                if (document.activeElement === firstFocusable) {
                    lastFocusable.focus();
                    e.preventDefault();
                }
            } else {
                // Tab
                if (document.activeElement === lastFocusable) {
                    firstFocusable.focus();
                    e.preventDefault();
                }
            }
        },

        /**
         * Close modal and restore focus
         * @param {HTMLElement} modal
         */
        closeModal: function (modal) {
            modal.remove();

            // Return focus to customize button
            const customizeBtn = document.querySelector('.unify-btn-flat');
            if (customizeBtn) {
                customizeBtn.focus();
                this.announce('Preferences dialog closed. Focus returned to customize button.');
            }
        },

        /**
         * Reject non-essential cookies and close banner
         */
        rejectAndClose: function () {
            const rejectBtn = document.querySelector('.unify-btn-reject');
            if (rejectBtn) {
                rejectBtn.click();
                this.announce('Rejected non-essential cookies. Banner closed.');
                this.restoreFocus();
            }
        },

        /**
         * Restore focus to previous element
         */
        restoreFocus: function () {
            if (this.previousFocus && this.previousFocus.focus) {
                setTimeout(() => {
                    this.previousFocus.focus();
                }, 100);
            }
        },

        /**
         * Enhance modal with accessibility features
         * @param {HTMLElement} modal
         */
        enhanceModal: function (modal) {
            if (!modal) return;

            // Add ARIA attributes
            const modalContent = modal.querySelector('[style*="background: white"]');
            if (modalContent) {
                modalContent.setAttribute('role', 'dialog');
                modalContent.setAttribute('aria-modal', 'true');
                modalContent.setAttribute('aria-labelledby', 'unify-modal-title');

                // Add ID to heading
                const heading = modalContent.querySelector('h2');
                if (heading) {
                    heading.id = 'unify-modal-title';
                }
            }

            // Store current focus
            this.previousFocus = document.activeElement;

            // Focus first interactive element
            setTimeout(() => {
                const firstFocusable = modal.querySelector(this.focusableElements);
                if (firstFocusable) {
                    firstFocusable.focus();
                }
                this.announce('Cookie preferences dialog opened. Select your preferences and save or cancel.');
            }, 100);
        }
    };

    /**
     * Initialize on DOM ready
     */
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => A11yEnhancements.init());
    } else {
        A11yEnhancements.init();
    }

    /**
     * Listen for custom modal events from banner.js
     * This is a more reliable hook than monkey-patching createElement
     */
    window.addEventListener('unifyModalOpened', function (e) {
        const { modal, modalContent } = e.detail;

        // Store current focus for restoration
        A11yEnhancements.previousFocus = document.activeElement;

        // Announce to screen readers
        A11yEnhancements.announce('Cookie preferences dialog opened. Select your preferences and save or cancel.');

        // Set up focus trap on the modal content (the actual dialog)
        A11yEnhancements.currentModal = modalContent;
        A11yEnhancements.currentModalOverlay = modal;
    });

    window.addEventListener('unifyModalClosed', function () {
        // Announce to screen readers
        A11yEnhancements.announce('Preferences dialog closed. Focus returned to customize button.');

        // Clear modal reference
        A11yEnhancements.currentModal = null;
        A11yEnhancements.currentModalOverlay = null;

        // Restore focus to the customize button
        const customizeBtn = document.querySelector('.unify-btn-flat');
        if (customizeBtn) {
            customizeBtn.focus();
        } else if (A11yEnhancements.previousFocus && A11yEnhancements.previousFocus.focus) {
            A11yEnhancements.previousFocus.focus();
        }
    });

    // Expose for external use
    window.UnifyA11y = A11yEnhancements;

})();

