/* public/assets/js/banner.js */
(function () {
    const COOKIE_NAME = 'unify_consent';

    function getCookie(name) {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
    }

    function setCookie(name, value, days) {
        const d = new Date();
        d.setTime(d.getTime() + (days * 24 * 60 * 60 * 1000));
        let expires = "expires=" + d.toUTCString();
        document.cookie = name + "=" + value + ";" + expires + ";path=/;SameSite=Lax";
    }

    /**
     * Script Re-enabler
     * Activates blocked scripts after user gives consent
     */
    const ScriptEnabler = {
        /**
         * Enable all blocked scripts that match the given categories
         * @param {Object} consent - Object with category: boolean pairs
         */
        enableScripts: function (consent) {
            const blockedScripts = document.querySelectorAll('script[data-blocked="true"]');

            blockedScripts.forEach(script => {
                const category = script.getAttribute('data-consent-category');

                // Check if user consented to this category
                if (consent[category]) {
                    this.executeScript(script);
                }
            });
        },

        /**
         * Execute a previously blocked script
         * @param {HTMLScriptElement} blockedScript - The blocked script element
         */
        executeScript: function (blockedScript) {
            // Create a new script element
            const newScript = document.createElement('script');

            // Copy attributes (except the blocking ones)
            Array.from(blockedScript.attributes).forEach(attr => {
                if (!['type', 'data-blocked', 'data-consent-category', 'data-src', 'data-original-src'].includes(attr.name)) {
                    newScript.setAttribute(attr.name, attr.value);
                }
            });

            // Set correct type
            newScript.type = 'text/javascript';

            // Handle external scripts
            const originalSrc = blockedScript.getAttribute('data-src') || blockedScript.getAttribute('data-original-src');
            if (originalSrc) {
                newScript.src = originalSrc;
                newScript.async = true;
            }

            // Handle inline scripts
            if (blockedScript.textContent && !originalSrc) {
                newScript.textContent = blockedScript.textContent;
            }

            // Remove the old blocked script
            blockedScript.parentNode.removeChild(blockedScript);

            // Insert the new executable script
            document.head.appendChild(newScript);
        }
    };

    const Banner = {
        init: function () {
            // Check if consent already exists
            const existingConsent = getCookie(COOKIE_NAME);

            if (existingConsent) {
                // User has already made a choice - enable scripts based on their preferences
                try {
                    const consent = JSON.parse(existingConsent);
                    // Scripts should already be loaded on page, but check for any that need re-enabling
                    // This handles edge cases where consent was given but page wasn't reloaded
                    if (typeof window.unifyConsentData !== 'undefined' && window.unifyConsentData.blockedCount > 0) {
                        ScriptEnabler.enableScripts(consent);
                    }
                } catch (e) {
                    console.warn('Unify: Could not parse consent cookie');
                }
                return; // Don't show banner
            }

            // No consent - show banner
            const banner = document.getElementById('unify-cookie-banner');
            if (banner) {
                banner.style.display = 'block';
            }

            this.bindEvents();
        },

        bindEvents: function () {
            // Accept All
            document.querySelectorAll('.unify-btn-accept').forEach(btn => {
                btn.addEventListener('click', () => this.saveConsent({
                    necessary: true, functional: true, analytics: true, marketing: true
                }));
            });

            // Reject All
            document.querySelectorAll('.unify-btn-reject').forEach(btn => {
                btn.addEventListener('click', () => this.saveConsent({
                    necessary: true, functional: false, analytics: false, marketing: false
                }));
            });

            // Customize - Show Modal
            document.querySelectorAll('.unify-btn-flat').forEach(btn => {
                btn.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.showCustomizeModal();
                });
            });
        },

        showCustomizeModal: function () {
            // Create modal overlay
            const modal = document.createElement('div');
            modal.id = 'unify-customize-modal';
            modal.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.7); z-index: 999999; display: flex; align-items: center; justify-content: center;';

            // Modal content with ARIA attributes for accessibility (WCAG 4.1.2)
            const modalContent = document.createElement('div');
            modalContent.id = 'unify-modal-content';
            modalContent.setAttribute('role', 'dialog');
            modalContent.setAttribute('aria-modal', 'true');
            modalContent.setAttribute('aria-labelledby', 'unify-modal-title');
            modalContent.setAttribute('aria-describedby', 'unify-modal-desc');
            modalContent.setAttribute('tabindex', '-1');
            modalContent.style.cssText = 'background: white; padding: 30px; border-radius: 8px; max-width: 500px; width: 90%; box-shadow: 0 10px 40px rgba(0,0,0,0.3);';

            modalContent.innerHTML = `
                <h2 id="unify-modal-title" style="margin-top: 0; font-size: 20px; color: #111827;">Customize Cookie Preferences</h2>
                <p id="unify-modal-desc" style="color: #6b7280; font-size: 14px; margin-bottom: 20px;">Choose which types of cookies you want to accept.</p>
                
                <div style="margin-bottom: 15px;">
                    <label style="display: flex; align-items: center; padding: 12px; background: #f9fafb; border-radius: 4px; cursor: not-allowed;">
                        <input type="checkbox" checked disabled style="margin-right: 10px;" aria-label="Necessary cookies - always enabled">
                        <div>
                            <strong style="display: block; font-size: 14px;">Necessary</strong>
                            <span style="font-size: 12px; color: #6b7280;">Required for the website to function</span>
                        </div>
                    </label>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <label style="display: flex; align-items: center; padding: 12px; background: #f9fafb; border-radius: 4px; cursor: pointer;">
                        <input type="checkbox" id="unify-functional" checked style="margin-right: 10px;" aria-label="Functional cookies - remember your preferences">
                        <div>
                            <strong style="display: block; font-size: 14px;">Functional</strong>
                            <span style="font-size: 12px; color: #6b7280;">Remember your preferences</span>
                        </div>
                    </label>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <label style="display: flex; align-items: center; padding: 12px; background: #f9fafb; border-radius: 4px; cursor: pointer;">
                        <input type="checkbox" id="unify-analytics" style="margin-right: 10px;" aria-label="Analytics cookies - help us improve our website">
                        <div>
                            <strong style="display: block; font-size: 14px;">Analytics</strong>
                            <span style="font-size: 12px; color: #6b7280;">Help us improve our website</span>
                        </div>
                    </label>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <label style="display: flex; align-items: center; padding: 12px; background: #f9fafb; border-radius: 4px; cursor: pointer;">
                        <input type="checkbox" id="unify-marketing" style="margin-right: 10px;" aria-label="Marketing cookies - personalized advertisements">
                        <div>
                            <strong style="display: block; font-size: 14px;">Marketing</strong>
                            <span style="font-size: 12px; color: #6b7280;">Personalized advertisements</span>
                        </div>
                    </label>
                </div>
                
                <div style="display: flex; gap: 10px; justify-content: flex-end;">
                    <button id="unify-modal-cancel" style="padding: 10px 20px; background: #f3f4f6; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 500;" aria-label="Cancel and close preferences">Cancel</button>
                    <button id="unify-modal-save" style="padding: 10px 20px; background: var(--unify-primary-color, #4F46E5); color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 500;" aria-label="Save cookie preferences">Save Preferences</button>
                </div>
            `;

            modal.appendChild(modalContent);
            document.body.appendChild(modal);

            // Dispatch custom event for a11y module to enhance (focus trap, etc.)
            window.dispatchEvent(new CustomEvent('unifyModalOpened', { detail: { modal, modalContent } }));

            // Focus the modal content for keyboard users
            setTimeout(() => modalContent.focus(), 100);

            // Handle cancel
            document.getElementById('unify-modal-cancel').addEventListener('click', () => {
                modal.remove();
                // Dispatch close event for a11y module
                window.dispatchEvent(new CustomEvent('unifyModalClosed'));
            });

            // Handle save
            document.getElementById('unify-modal-save').addEventListener('click', () => {
                const preferences = {
                    necessary: true,
                    functional: document.getElementById('unify-functional').checked,
                    analytics: document.getElementById('unify-analytics').checked,
                    marketing: document.getElementById('unify-marketing').checked
                };
                modal.remove();
                // Dispatch close event for a11y module
                window.dispatchEvent(new CustomEvent('unifyModalClosed'));
                this.saveConsent(preferences);
            });

            // Close on overlay click
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.remove();
                    window.dispatchEvent(new CustomEvent('unifyModalClosed'));
                }
            });
        },

        saveConsent: function (preferences) {
            // 1. Set Cookie
            setCookie(COOKIE_NAME, JSON.stringify(preferences), uwpcBannerParams.expiry);

            // 2. Hide Banner
            const banner = document.getElementById('unify-cookie-banner');
            if (banner) banner.style.display = 'none';

            // 3. Send to Server (Async)
            if (uwpcBannerParams.ajax_url) {
                const data = new FormData();
                data.append('action', 'unify_save_consent');
                data.append('nonce', uwpcBannerParams.nonce);

                // Add preferences to form data
                for (const [key, value] of Object.entries(preferences)) {
                    if (value) data.append(`consent[]`, key);
                }

                if (navigator.sendBeacon) {
                    navigator.sendBeacon(uwpcBannerParams.ajax_url, data);
                } else {
                    fetch(uwpcBannerParams.ajax_url, { method: 'POST', body: data });
                }
            }

            // 4. Enable blocked scripts based on consent
            ScriptEnabler.enableScripts(preferences);

            // 5. Dispatch custom event for other scripts to listen to
            window.dispatchEvent(new CustomEvent('unifyConsentUpdated', {
                detail: preferences
            }));

            // 6. Optional: Reload page if there were many blocked scripts
            // This ensures all scripts load in correct order
            if (typeof window.unifyConsentData !== 'undefined' && window.unifyConsentData.blockedCount > 3) {
                // Many scripts blocked - safer to reload
                window.location.reload();
            }
        }
    };

    // Initialize when DOM is ready
    document.addEventListener('DOMContentLoaded', () => Banner.init());

    // Expose for external access
    window.UnifyCookieConsent = {
        hasConsent: function (category) {
            const consent = getCookie(COOKIE_NAME);
            if (!consent) return false;
            try {
                const parsed = JSON.parse(consent);
                return !!parsed[category];
            } catch (e) {
                return false;
            }
        },
        getConsent: function () {
            const consent = getCookie(COOKIE_NAME);
            if (!consent) return null;
            try {
                return JSON.parse(consent);
            } catch (e) {
                return null;
            }
        },
        showBanner: function () {
            const banner = document.getElementById('unify-cookie-banner');
            if (banner) banner.style.display = 'block';
        }
    };
})();
