<?php
/**
 * WCAG 2.2 Accessibility Tests
 * 
 * Tests for accessibility compliance in the cookie banner
 * 
 * @package Unify_Compliance
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Unify_A11y_Tests {
    
    /**
     * Get the rendered banner HTML for testing
     * 
     * @return string
     */
    private function get_banner_html() {
        ob_start();
        
        // Set up test variables
        $banner_text = 'We use cookies to improve your experience.';
        $position = 'bottom';
        $theme = 'light';
        $primary_color = '#4F46E5';
        $accept_text = 'Accept All';
        $reject_text = 'Reject All';
        $customize_text = 'Customize';
        
        // Render the banner
        include Unify_PLUGIN_DIR . 'public/views/cookie-banner.php';
        
        return ob_get_clean();
    }
    
    /**
     * Test that banner has dialog role (WCAG 4.1.2)
     */
    public function test_banner_has_dialog_role() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'role="dialog"') !== false) {
            fwrite(STDOUT, "✅ PASS: Banner has role='dialog'\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Banner missing role='dialog'\n");
            return false;
        }
    }
    
    /**
     * Test that banner has aria-modal attribute
     */
    public function test_banner_has_aria_modal() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'aria-modal="true"') !== false) {
            fwrite(STDOUT, "✅ PASS: Banner has aria-modal='true'\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Banner missing aria-modal='true'\n");
            return false;
        }
    }
    
    /**
     * Test that banner has aria-labelledby (WCAG 4.1.2)
     */
    public function test_banner_has_aria_labelledby() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'aria-labelledby=') !== false) {
            fwrite(STDOUT, "✅ PASS: Banner has aria-labelledby\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Banner missing aria-labelledby\n");
            return false;
        }
    }
    
    /**
     * Test that banner has aria-describedby
     */
    public function test_banner_has_aria_describedby() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'aria-describedby=') !== false) {
            fwrite(STDOUT, "✅ PASS: Banner has aria-describedby\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Banner missing aria-describedby\n");
            return false;
        }
    }
    
    /**
     * Test that title has correct ID
     */
    public function test_title_has_id() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'id="unify-banner-title"') !== false) {
            fwrite(STDOUT, "✅ PASS: Title has id='unify-banner-title'\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Title missing id='unify-banner-title'\n");
            return false;
        }
    }
    
    /**
     * Test that description has correct ID
     */
    public function test_description_has_id() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'id="unify-banner-description"') !== false) {
            fwrite(STDOUT, "✅ PASS: Description has id='unify-banner-description'\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Description missing id='unify-banner-description'\n");
            return false;
        }
    }
    
    /**
     * Test that buttons have aria-label (WCAG 4.1.2)
     */
    public function test_buttons_have_accessible_names() {
        $html = $this->get_banner_html();
        
        // Count button elements
        $button_count = substr_count($html, '<button');
        
        // Count aria-label attributes on buttons
        preg_match_all('/<button[^>]*aria-label=/', $html, $matches);
        $aria_label_count = count($matches[0]);
        
        if ($button_count > 0 && $aria_label_count === $button_count) {
            fwrite(STDOUT, "✅ PASS: All $button_count buttons have aria-label\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Only $aria_label_count of $button_count buttons have aria-label\n");
            return false;
        }
    }
    
    /**
     * Test that privacy link has accessible label
     */
    public function test_privacy_link_has_label() {
        $html = $this->get_banner_html();
        
        if (strpos($html, 'aria-label=') !== false && strpos($html, '<a href=') !== false) {
            fwrite(STDOUT, "✅ PASS: Privacy link has aria-label\n");
            return true;
        } else {
            fwrite(STDOUT, "⚠️  WARNING: Privacy link may be missing aria-label (may not be rendered in test)\n");
            return true; // Not a hard fail since link might not be present
        }
    }
    
    /**
     * Test WCAG mode data attribute when premium
     */
    public function test_wcag_mode_attribute() {
        if (!function_exists('unify_is_premium')) {
            fwrite(STDOUT, "⚠️  SKIP: unify_is_premium() function not available\n");
            return true;
        }
        
        if (!unify_is_premium()) {
            fwrite(STDOUT, "⚠️  SKIP: Premium not active, WCAG mode not tested\n");
            return true;
        }
        
        $html = $this->get_banner_html();
        
        if (strpos($html, 'data-wcag-mode="true"') !== false) {
            fwrite(STDOUT, "✅ PASS: Banner has data-wcag-mode='true' when premium\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ FAIL: Banner missing data-wcag-mode='true' despite premium\n");
            return false;
        }
    }
    
    /**
     * Run all tests
     * 
     * @return array Results
     */
    public function run_all() {
        fwrite(STDOUT, "\n=== WCAG 2.2 Accessibility Tests ===\n\n");
        
        $results = [];
        
        $results['dialog_role'] = $this->test_banner_has_dialog_role();
        $results['aria_modal'] = $this->test_banner_has_aria_modal();
        $results['aria_labelledby'] = $this->test_banner_has_aria_labelledby();
        $results['aria_describedby'] = $this->test_banner_has_aria_describedby();
        $results['title_id'] = $this->test_title_has_id();
        $results['description_id'] = $this->test_description_has_id();
        $results['button_labels'] = $this->test_buttons_have_accessible_names();
        $results['link_label'] = $this->test_privacy_link_has_label();
        $results['wcag_mode'] = $this->test_wcag_mode_attribute();
        
        // Summary
        $passed = count(array_filter($results));
        $total = count($results);
        
        fwrite(STDOUT, "\n=== Summary ===\n");
        fwrite(STDOUT, "Passed: $passed / $total tests\n");
        
        if ($passed === $total) {
            fwrite(STDOUT, "✅ All accessibility tests passed!\n\n");
            return true;
        } else {
            fwrite(STDOUT, "❌ Some tests failed. Review output above.\n\n");
            return false;
        }
    }
}
