<?php
/**
 * Plugin Name: Unify Compliance
 * Plugin URI: https://arwendigital.net/unify
 * Description: One Dashboard. All Regulations. Manage GDPR, NIS2, EU AI Act, and CCPA in a single interface.
 * Version: 1.0.3
 * Author: Unify Team
 * Author URI: https://arwendigital.net
 * Text Domain: unify-compliance
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'Unify_VERSION', '1.0.4' );
define( 'Unify_PLUGIN_FILE', __FILE__ );
define( 'Unify_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'Unify_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Bootstrap Plugin
 */
require_once Unify_PLUGIN_DIR . 'includes/class-unify-core.php';
require_once Unify_PLUGIN_DIR . 'includes/class-unify-module-manager.php';
require_once Unify_PLUGIN_DIR . 'includes/class-unify-reports.php';
require_once Unify_PLUGIN_DIR . 'includes/class-unify-dashboard-data.php';
require_once Unify_PLUGIN_DIR . 'includes/class-unify-license.php';
require_once Unify_PLUGIN_DIR . 'includes/nis2/class-unify-log-cleanup.php';
require_once Unify_PLUGIN_DIR . 'includes/nis2/class-unify-policy-manager.php';
require_once Unify_PLUGIN_DIR . 'premium/class-unify-vulnerability-scanner.php';
Unify_Core::run();
Unify_Log_Cleanup::init();

register_activation_hook( __FILE__, array( 'Unify_Log_Cleanup', 'schedule_events' ) );
register_deactivation_hook( __FILE__, array( 'Unify_Log_Cleanup', 'unschedule_events' ) );

// License cron jobs
register_activation_hook( __FILE__, array( 'Unify_License', 'schedule_validation_cron' ) );
register_deactivation_hook( __FILE__, array( 'Unify_License', 'unschedule_validation_cron' ) );
add_action( 'unify_daily_license_check', array( 'Unify_License', 'run_daily_validation' ) );

/**
 * Helper to check premium status
 */
function unify_is_premium() {
    return Unify_License::is_active();
}


/**
 * Enqueue Admin Assets
 */
function unify_enqueue_admin_assets( $hook ) {
	if ( strpos( $hook, 'unify' ) === false ) {
		return;
	}

	wp_enqueue_style( 'unify-admin-style', Unify_PLUGIN_URL . 'admin/assets/css/admin.css', array(), Unify_VERSION );
	wp_enqueue_script( 'unify-admin-script', Unify_PLUGIN_URL . 'admin/assets/js/admin.js', array( 'jquery' ), Unify_VERSION, true );
	
	// Enqueue Chart.js on reports pages
	if ( isset( $_GET['page'] ) && $_GET['page'] === 'unify-reports' ) {
		wp_enqueue_script( 'chartjs', Unify_PLUGIN_URL . 'admin/assets/js/chart.min.js', array(), '4.4.1', true );
	}
	
	wp_localize_script( 'unify-admin-script', 'unify_vars', array(
		'nonce' => wp_create_nonce( 'unify_admin_nonce' ),
		'nis2_nonce' => wp_create_nonce( 'unify_save_nis2_settings' ),
		'ai_nonce' => wp_create_nonce( 'unify_save_ai_system' ),
        'modules_nonce' => wp_create_nonce( 'unify_save_modules' ),
        'reports_nonce' => wp_create_nonce( 'unify_reports_nonce' ),
        'dashboard_nonce' => wp_create_nonce( 'unify_dashboard_nonce' ),
        'license_nonce' => wp_create_nonce( 'unify_license_nonce' ),
	));
	
	// Add inline script to handle Premium link opening in new tab
	wp_add_inline_script( 'unify-admin-script', "
		document.addEventListener('DOMContentLoaded', function() {
			var premiumLink = document.querySelector('a[href=\"https://arwendigital.net/wordpress-plugin/unify-compliance\"]');
			if (premiumLink) {
				premiumLink.setAttribute('target', '_blank');
				premiumLink.setAttribute('rel', 'noopener noreferrer');
			}
		});
	" );
}
add_action( 'admin_enqueue_scripts', 'unify_enqueue_admin_assets' );

/**
 * Register Admin Menu
 */
function unify_register_admin_menu() {
	add_menu_page(
		'Unify Compliance',
		'Unify',
		'manage_options',
		'unify-compliance',
		'unify_render_dashboard',
		'dashicons-shield',
		30
	);

	add_submenu_page(
		'unify-compliance',
		'Dashboard',
		'Dashboard',
		'manage_options',
		'unify-compliance',
		'unify_render_dashboard'
	);


	add_submenu_page(
		'unify-compliance',
		'Modules',
		'Modules',
		'manage_options',
		'unify-modules',
		'unify_render_modules'
	);

    if ( Unify_Module_Manager::is_module_active( 'gdpr' ) ) {
        add_submenu_page(
            'unify-compliance',
            'GDPR',
            'GDPR',
            'manage_options',
            'unify-gdpr',
            'unify_render_gdpr'
        );
    }

    if ( Unify_Module_Manager::is_module_active( 'nis2' ) ) {
        add_submenu_page(
            'unify-compliance',
            'NIS2',
            'NIS2',
            'manage_options',
            'unify-nis2',
            'unify_render_nis2'
        );
    }
    
    if ( Unify_Module_Manager::is_module_active( 'ai-act' ) ) { // Removed unify_is_premium() check for visibility so upgrade prompts can work if implemented that way, or keep strict. Keeping strict per logic.
        add_submenu_page(
            'unify-compliance',
            'AI Act',
            'AI Act',
            'manage_options',
            'unify-ai-act',
            'unify_render_ai_act'
        );
    }

	add_submenu_page(
		'unify-compliance',
		'Reports',
		'Reports',
		'manage_options',
		'unify-reports',
		'unify_render_reports'
	);

	add_submenu_page(
		'unify-compliance',
		'Settings',
		'Settings',
		'manage_options',
		'unify-settings',
		'unify_render_settings'
	);

	// Premium upgrade link - only show for free tier users
	if ( ! unify_is_premium() ) {
		global $submenu;
		$submenu['unify-compliance'][] = array(
			'<span style="color: #f59e0b; font-weight: 600;">⭐ Premium</span>',
			'manage_options',
			'https://arwendigital.net/wordpress-plugin/unify-compliance'
		);
	}
}
add_action( 'admin_menu', 'unify_register_admin_menu' );

/**
 * View Render Functions
 */
function unify_get_view( $view_name ) {
    $file_path = Unify_PLUGIN_DIR . 'admin/views/' . $view_name . '.php';
    if ( file_exists( $file_path ) ) {
        // Enqueue media if needed, etc.
        include Unify_PLUGIN_DIR . 'admin/partials/header.php';
        include $file_path;
        include Unify_PLUGIN_DIR . 'admin/partials/footer.php';
    } else {
        echo '<div class="wrap"><h1>View not found (' . esc_html($view_name) . ')</h1></div>';
    }
}

function unify_render_dashboard() {
    // Check for welcome wizard param
    if ( isset($_GET['setup']) ) {
        if ( $_GET['setup'] === 'checklist' ) {
            unify_get_view('checklist');
        } else {
            unify_get_view('welcome-wizard');
        }
    } else {
	    unify_get_view('dashboard');
    }
}

function unify_render_nis2() {
    unify_get_view('nis2/dashboard');
}

function unify_render_modules() {
	unify_get_view('module-selector');
}

function unify_render_gdpr() {
    $tab = isset($_GET['tab']) ? $_GET['tab'] : 'banner';
    if ($tab === 'banner') unify_get_view('gdpr/cookie-banner');
    elseif ($tab === 'queue') unify_get_view('gdpr/dsar-queue');
    elseif ($tab === 'form') unify_get_view('gdpr/dsar-form-settings');
    elseif ($tab === 'scanner') unify_get_view('gdpr/cookie-scanner');
    elseif ($tab === 'automation') unify_get_view('gdpr/dsar-automation');
    else unify_get_view('gdpr/cookie-banner');
}

function unify_render_ai_act() {
    unify_get_view('ai-act/dashboard');
}

function unify_render_reports() {
    $tab = isset($_GET['tab']) ? $_GET['tab'] : 'overview';
    if ($tab === 'overview') unify_get_view('reports/overview');
    elseif ($tab === 'consent') unify_get_view('reports/consent-analytics');
    elseif ($tab === 'security') unify_get_view('reports/security-metrics');
    else unify_get_view('reports/overview');
}

function unify_render_settings() {
    $tab = isset($_GET['tab']) ? $_GET['tab'] : 'license';
    if ($tab === 'license') unify_get_view('settings/license');
    else unify_get_view('settings/license');
}
